<?php

namespace UltimatePostKitPro\Modules\SoftTimeline\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Text_Stroke;
use Elementor\Group_Control_Background;
use UltimatePostKitPro\Utils;

use UltimatePostKit\Traits\Global_Widget_Controls;
use UltimatePostKit\Traits\Global_Widget_Functions;
use UltimatePostKit\Includes\Controls\GroupQuery\Group_Control_Query;
use WP_Query;

if (!defined('ABSPATH')) {
	exit;
} // Exit if accessed directly

class Soft_Timeline extends Group_Control_Query {

	use Global_Widget_Controls;
	use Global_Widget_Functions;

	private $_query = null;

	public function get_name() {
		return 'upk-soft-timeline';
	}

	public function get_title() {
		return BDTUPK . esc_html__('Soft Timeline', 'ultimate-post-kit-pro');
	}

	public function get_icon() {
		return 'upk-widget-icon upk-icon-soft-timeline';
	}

	public function get_categories() {
		return ['ultimate-post-kit-pro'];
	}

	public function get_keywords() {
		return ['post', 'blog', 'recent', 'news', 'soft', 'timeline'];
	}

	public function get_style_depends() {
		if ($this->upk_is_edit_mode()) {
			return ['upk-all-styles-pro'];
		} else {
			return ['upk-font', 'upk-soft-timeline'];
		}
	}

	public function get_custom_help_url() {
		return 'https://youtu.be/5scXg5bsGDc';
	}

	public function get_query() {
		return $this->_query;
	}

	public function has_widget_inner_wrapper(): bool {
        return ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
    }
	protected function is_dynamic_content(): bool {
		return true;
	}

	protected function register_controls() {
		$this->start_controls_section(
			'section_content_layout',
			[
				'label' => esc_html__('Layout', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'content_position',
			[
				'label'   => __('Position', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::CHOOSE,
				'default' => 'center',
				'options' => [
					'left'   => [
						'title' => __('Left', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-h-align-left',
					],
					'center' => [
						'title' => __('Center', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-h-align-center',
					],
					'right'  => [
						'title' => __('Right', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-h-align-right',
					],
				],
				'prefix_class' => 'upk-timeline-style-',
				'render_type' => 'template',
				'toggle' => false
			]
		);

		//Global Title Controls
		$this->register_title_controls();

		$this->add_control(
			'show_excerpt',
			[
				'label'   => esc_html__('Show Text', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'excerpt_length',
			[
				'label'       => esc_html__('Text Limit', 'ultimate-post-kit-pro'),
				'description' => esc_html__('It\'s just work for main content, but not working with excerpt. If you set 0 so you will get full main content.', 'ultimate-post-kit-pro'),
				'type'        => Controls_Manager::NUMBER,
				'default'     => 20,
				'condition'   => [
					'show_excerpt' => 'yes'
				],
			]
		);

		$this->add_control(
			'strip_shortcode',
			[
				'label'     => esc_html__('Strip Shortcode', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'show_category',
			[
				'label'   => esc_html__('Show Category', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SWITCHER,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'show_post_format',
			[
				'label'   => esc_html__('Show Post Formate', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
			'show_author',
			[
				'label'   => esc_html__('Show Author', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'show_comments',
			[
				'label'   => esc_html__('Show Comments', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		$this->add_control(
			'meta_separator',
			[
				'label'       => __('Separator', 'ultimate-post-kit-pro'),
				'type'        => Controls_Manager::TEXT,
				'default'     => '//',
				'label_block' => false,
			]
		);

		//Global Date Controls
		$this->register_date_controls();

		//Global Reading Time Controls
		$this->register_reading_time_controls();

		$this->add_control(
			'show_pagination',
			[
				'label'     => esc_html__('Pagination', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SWITCHER,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'global_link',
			[
				'label'        => __('Item Wrapper Link', 'ultimate-post-kit-pro'),
				'type'         => Controls_Manager::SWITCHER,
				'prefix_class' => 'upk-global-link-',
				'description'  => __('Be aware! When Item Wrapper Link activated then title link and read more link will not work', 'ultimate-post-kit-pro'),
			]
		);

		$this->end_controls_section();

		// Query Settings
		$this->start_controls_section(
			'section_post_query_builder',
			[
				'label' => __('Query', 'ultimate-post-kit-pro'),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'item_limit',
			[
				'label'   => esc_html__('Item Limit', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SLIDER,
				'range'   => [
					'px' => [
						'min' => 1,
						'max' => 20,
					],
				],
				'default' => [
					'size' => 6,
				],
				'separator' => 'after'
			]
		);

		$this->register_query_builder_controls();

		$this->end_controls_section();

		//Style
		$this->start_controls_section(
			'upk_section_style',
			[
				'label' => esc_html__('Items', 'ultimate-post-kit-pro'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'timeline_item_background',
				'selector'  => '{{WRAPPER}} .upk-soft-timeline .upk-soft-content-wrap',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'           => 'timeline_item_border',
				'label'          => __('Border', 'ultimate-post-kit-pro'),
				'fields_options' => [
					'border' => [
						'default' => 'solid',
					],
					'width'  => [
						'default' => [
							'top'      => '1',
							'right'    => '1',
							'bottom'   => '1',
							'left'     => '1',
							'isLinked' => false,
						],
					],
					'color'  => [
						'default' => '#f2f2f2',
					],
				],
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-soft-content-wrap',
			]
		);

		$this->add_responsive_control(
			'timeline_item_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-soft-timeline .upk-soft-content-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'timeline_item_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-soft-timeline .upk-soft-content-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'timeline_item_shadow',
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-soft-content-wrap',
			]
		);

		$this->add_control(
			'item_arrows_color',
			[
				'label'     => esc_html__('Indicator Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}}.upk-timeline-style-center .upk-soft-item:nth-child(even) .upk-soft-content-wrap::after, {{WRAPPER}}.upk-timeline-style-right .upk-soft-content-wrap::after' => 'border-color: transparent transparent transparent {{VALUE}};',
					'{{WRAPPER}}.upk-timeline-style-center .upk-soft-item:nth-child(odd) .upk-soft-content-wrap::before, {{WRAPPER}}.upk-timeline-style-left .upk-soft-content-wrap::before, {{WRAPPER}}.upk-timeline-style-center .upk-soft-content-wrap::before' => 'border-color: transparent {{VALUE}} transparent transparent;'
				],
			]
		);

		$this->start_controls_tabs('tabs_line_dots_style');

		$this->start_controls_tab(
			'tab_line_dots_normal',
			[
				'label' => esc_html__('LINE / DOTS', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'item_line_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-soft-wrap::before, {{WRAPPER}} .upk-soft-timeline .upk-soft-item::before' => 'background: {{VALUE}};'
				],
			]
		);

		$this->add_responsive_control(
			'item_border_width',
			[
				'label'     => esc_html__('Line Border Width', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0.1,
						'max'  => 5,
						'step' => 0.1
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-soft-wrap::before' => 'width: {{SIZE}}px;'
				],
			]
		);

		$this->add_responsive_control(
			'item_dots_border_radius',
			[
				'label'      => esc_html__('Dots Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-soft-timeline .upk-soft-item::before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_start_end',
			[
				'label' => esc_html__('START / END', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'item_start_end_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-soft-start-end' => 'color: {{VALUE}};'
				],
			]
		);

		$this->add_control(
			'item_start_end_bg_color',
			[
				'label'     => esc_html__('Background', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-soft-start-end' => 'background: {{VALUE}};'
				],
			]
		);

		$this->add_responsive_control(
			'item_start_end_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-soft-timeline .upk-soft-start-end' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_start_end_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-soft-start-end',
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_title',
			[
				'label'     => esc_html__('Title', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_title' => 'yes',
				],
			]
		);

		$this->add_control(
			'title_style',
			[
				'label'   => esc_html__('Style', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SELECT,
				'default' => 'underline',
				'options' => [
					'underline'        => esc_html__('Underline', 'ultimate-post-kit-pro'),
					'middle-underline' => esc_html__('Middle Underline', 'ultimate-post-kit-pro'),
					'overline'         => esc_html__('Overline', 'ultimate-post-kit-pro'),
					'middle-overline'  => esc_html__('Middle Overline', 'ultimate-post-kit-pro'),
				],
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_hover_color',
			[
				'label'     => esc_html__('Hover Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-title a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-title',
			]
		);

		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name'     => 'title_text_shadow',
				'label'    => __('Text Shadow', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-title',
			]
		);
		
		$this->add_group_control(
			Group_Control_Text_Stroke::get_type(),
			[
				'name'     => 'title_text_stroke',
				'label'    => __('Text Stroke', 'ultimate-post-kit-pro') . BDTUPK_NC,
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-title a',
			]
		);

		$this->add_responsive_control(
			'title_spacing',
			[
				'label'     => esc_html__('Spacing', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_text',
			[
				'label'     => esc_html__('Text', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'text_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-soft-desc' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'text_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-soft-desc',
			]
		);

		$this->add_responsive_control(
			'text_margin',
			[
				'label'      => __('Margin', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-soft-timeline .upk-soft-desc' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_meta',
			[
				'label'      => esc_html__('Meta', 'ultimate-post-kit-pro'),
				'tab'        => Controls_Manager::TAB_STYLE,
				'conditions' => [
					'relation' => 'or',
					'terms'    => [
						[
							'name'  => 'show_author',
							'value' => 'yes'
						],
						[
							'name'  => 'show_comments',
							'value' => 'yes'
						]
					]
				],
			]
		);

		$this->add_control(
			'meta_color',
			[
				'label'     => esc_html__('Text Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-timeline-meta, {{WRAPPER}} .upk-soft-timeline .upk-timeline-meta .upk-author-name-wrap .upk-author-name' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_hover_color',
			[
				'label'     => esc_html__('Text Hover Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-timeline-meta .upk-author-name-wrap .upk-author-name:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'meta_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-timeline-meta',
			]
		);

		$this->add_responsive_control(
			'meta_spacing',
			[
				'label'     => esc_html__('Spacing', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-timeline-meta' => 'padding-top: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'meta_space_between',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-timeline-meta > div:before' => 'margin: 0 {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_date',
			[
				'label'      => esc_html__('Date', 'ultimate-post-kit-pro'),
				'tab'        => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_date' => 'yes'
				],
			]
		);

		$this->add_control(
			'date_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-post-time, {{WRAPPER}} .upk-soft-timeline .upk-soft-date-wrap' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'date_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-post-time, {{WRAPPER}} .upk-soft-timeline .upk-soft-date-wrap',
			]
		);

		$this->add_responsive_control(
			'date_space_between',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-post-time' => 'margin-left: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_category',
			[
				'label'     => esc_html__('Category', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_category' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'category_bottom_spacing',
			[
				'label'   => esc_html__('Spacing', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-timeline-category' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs('tabs_category_style');

		$this->start_controls_tab(
			'tab_category_normal',
			[
				'label' => esc_html__('Normal', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'category_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-timeline-category a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'category_background',
				'selector'  => '{{WRAPPER}} .upk-soft-timeline .upk-timeline-category a',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'        => 'category_border',
				'selector'    => '{{WRAPPER}} .upk-soft-timeline .upk-timeline-category a',
			]
		);

		$this->add_responsive_control(
			'category_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-soft-timeline .upk-timeline-category a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-soft-timeline .upk-timeline-category a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_spacing',
			[
				'label'   => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-timeline-category a+a' => 'margin-left: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'category_shadow',
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-timeline-category a',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'category_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-timeline-category a',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_category_hover',
			[
				'label' => esc_html__('Hover', 'ultimate-post-kit-pro'),
				'condition' => [
					'show_category' => 'yes'
				]
			]
		);

		$this->add_control(
			'category_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-timeline-category a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'category_hover_background',
				'selector'  => '{{WRAPPER}} .upk-soft-timeline .upk-timeline-category a:hover',
			]
		);

		$this->add_control(
			'category_hover_border_color',
			[
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'condition' => [
					'category_border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-timeline-category a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		//Post Format
		$this->start_controls_section(
			'section_style_post_format',
			[
				'label'     => esc_html__('Post Format', 'ultimate-post-kit-pro') . BDTUPK_NC,
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_post_format' => 'yes'
				],
			]
		);

		$this->start_controls_tabs('tabs_post_format_style');
		$this->start_controls_tab(
			'tab_post_format_normal',
			[
				'label' => esc_html__('Normal', 'ultimate-post-kit-pro'),
			]
		);
		
		$this->add_control(
			'item_formate_icon_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-post-format a' => 'color: {{VALUE}};'
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'item_formate_icon_background',
				'selector'  => '{{WRAPPER}} .upk-soft-timeline .upk-post-format a',
			]
		);
		
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'        => 'item_formate_icon_border',
				'selector'    => '{{WRAPPER}} .upk-soft-timeline .upk-post-format a',
				'separator' => 'before'
			]
		);
		
		$this->add_responsive_control(
			'item_formate_icon_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .upk-soft-timeline .upk-post-format a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]
		);
		
		$this->add_responsive_control(
			'item_formate_icon_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .upk-soft-timeline .upk-post-format a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]
		);
		
		$this->add_responsive_control(
			'item_formate_icon_margin',
			[
				'label'      => esc_html__('Margin', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .upk-soft-timeline .upk-post-format' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_formate_icon_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-post-format a',
			]
		);
		
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'item_formate_icon_shadow',
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-post-format a',
			]
		);
		
		$this->end_controls_tab();
		$this->start_controls_tab(
			'tab_post_format_hover',
			[
				'label' => esc_html__('Hover', 'ultimate-post-kit-pro'),
			]
		);
		
		$this->add_control(
			'item_formate_icon_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-post-format a:hover' => 'color: {{VALUE}};'
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'item_formate_icon_hover_background',
				'selector'  => '{{WRAPPER}} .upk-soft-timeline .upk-post-format a:hover',
			]
		);
		
		$this->add_control(
			'item_formate_icon_hover_border_color',
			[
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'condition' => [
					'item_formate_icon_border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .upk-soft-timeline .upk-post-format a:hover' => 'border-color: {{VALUE}};'
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'item_formate_icon_hover_shadow',
				'selector' => '{{WRAPPER}} .upk-soft-timeline .upk-post-format a:hover',
			]
		);

		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();

		//Global Pagination Controls
		$this->register_pagination_controls();
	}

	/**
	 * Main query render for this widget
	 * @param $posts_per_page number item query limit
	 */
	public function query_posts($posts_per_page) {

		$default = $this->getGroupControlQueryArgs();
		if ($posts_per_page) {
			$args['posts_per_page'] = $posts_per_page;
			$args['paged']  = max(1, get_query_var('paged'), get_query_var('page'));
		}
		$args         = array_merge($default, $args);
		$this->_query = new WP_Query($args);
	}

	public function render_image($image_id, $size) {

		if (!$this->get_settings('show_image')) {
			return;
		}

		$placeholder_image_src = Utils::get_placeholder_image_src();
		$image_src             = wp_get_attachment_image_src($image_id, $size);

		if (!$image_src) {
			$image_src = $placeholder_image_src;
		} else {
			$image_src = $image_src[0];
		}

		?>

		<div class="upk-timeline-image-wrapper">
			<img class="upk-img" src="<?php echo esc_url($image_src); ?>" alt="<?php echo esc_html(get_the_title()); ?>">
		</div>
		<?php
	}

	public function render_category() {

		if (!$this->get_settings('show_category')) {
			return;
		}
	?>
		<div class="upk-timeline-category">
			<?php echo upk_get_category($this->get_settings('posts_source')); ?>
		</div>
	<?php
	}

	public function render_author() {

		if (!$this->get_settings('show_author')) {
			return;
		}
	?>
		<div class="upk-author-name-wrap">
			<span class="upk-by"><?php echo esc_html_x('by', 'Frontend', 'ultimate-post-kit-pro') ?></span>
			<a class="upk-author-name" href="<?php echo get_author_posts_url(get_the_author_meta('ID')) ?>">
				<?php echo get_the_author() ?>
			</a>
		</div>
	<?php
	}

	public function render_excerpt($excerpt_length) {

		if (!$this->get_settings('show_excerpt')) {
			return;
		}
		$strip_shortcode = $this->get_settings_for_display('strip_shortcode');
	?>
		<div class="upk-soft-desc">
			<?php
			if (has_excerpt()) {
				the_excerpt();
			} else {
				echo ultimate_post_kit_custom_excerpt($excerpt_length, $strip_shortcode);
			}
			?>
		</div>

	<?php
	}

	public function render_comments($id = 0) {

		if (!$this->get_settings('show_comments')) {
			return;
		}
	?>

		<div class="upk-timeline-comments">
			<?php echo get_comments_number($id) ?>
			<?php echo esc_html__('Comments', 'ultimate-post-kit-pro') ?>
		</div>

	<?php
	}

	public function render_date() {
		$settings = $this->get_settings_for_display();


		if (!$this->get_settings('show_date')) {
			return;
		}

	?>
		<div class="upk-timeline-date upk-flex upk-flex-middle">
			<div class="upk-soft-date-wrap">
				<?php if ($settings['human_diff_time'] == 'yes') {
					echo ultimate_post_kit_post_time_diff(($settings['human_diff_time_short'] == 'yes') ? 'short' : '');
				} else {
					echo get_the_date();
				} ?>
			</div>
			<?php if ($settings['show_time']) : ?>
				<div class="upk-post-time">
					<i class="upk-icon-clock" aria-hidden="true"></i>
					<?php echo get_the_time(); ?>
				</div>
			<?php endif; ?>
		</div>

	<?php
	}

	public function render_post_grid_item($post_id, $excerpt_length) {
		$settings = $this->get_settings_for_display();

		if ('yes' == $settings['global_link']) {

			$this->add_render_attribute('timeline-item', 'onclick', "window.open('" . esc_url(get_permalink()) . "', '_self')", true);
		}
		$this->add_render_attribute('timeline-item', 'class', 'upk-soft-item', true);

	?>
		<div <?php $this->print_render_attribute_string('timeline-item'); ?>>
			<div class="upk-inner-item">
				<?php $this->render_date(); ?>
			</div>
			<div class="upk-inner-item">
				<div class="upk-soft-content-wrap">
					<?php $this->render_post_format(); ?>
					<div class="upk-soft-text">
						<?php $this->render_category(); ?>
						<?php $this->render_title(substr($this->get_name(), 4)); ?>
						<?php $this->render_excerpt($excerpt_length); ?>

						<?php if ($settings['show_author'] or $settings['show_comments'] or $settings['show_reading_time']) : ?>
						<div class="upk-timeline-meta">
							<?php $this->render_author(); ?>

							<?php if ($settings['show_comments']) : ?>
								<div data-separator="<?php echo esc_html($settings['meta_separator']); ?>">
								<?php $this->render_comments($post_id); ?>
								</div>
							<?php endif; ?>

							<?php if ('yes' === $settings['show_reading_time']) : ?>
								<div class="upk-reading-time" data-separator="<?php echo esc_html($settings['meta_separator']); ?>">
									<?php echo ultimate_post_kit_reading_time(get_the_content(), $settings['avg_reading_speed']); ?>
								</div>
							<?php endif; ?>
						</div>
						<?php endif; ?>
						
					</div>
				</div>
			</div>
		</div>
	<?php
	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		$limit = $settings['item_limit']['size'];

		$this->query_posts($limit);

		$wp_query = $this->get_query();

		if (!$wp_query->found_posts) {
			return;
		}

	?>

		<div class="upk-soft-timeline">
			<div class="upk-soft-wrap">
				<div class="upk-soft-start-end upk-soft-start"><?php echo esc_html__('start', 'ultimate-post-kit-pro') ?></div>
				<div class="upk-soft-start-end upk-soft-end"><?php echo esc_html__('end', 'ultimate-post-kit-pro') ?></div>
				<div class="upk-soft-grid-wrap">

					<?php

					while ($wp_query->have_posts()) :
						$wp_query->the_post();

					?>

						<?php $this->render_post_grid_item(get_the_ID(), $settings['excerpt_length']); ?>

					<?php endwhile; ?>
				</div>
			</div>
		</div>

		<?php

		if ($settings['show_pagination']) { ?>
			<div class="ep-pagination">
				<?php ultimate_post_kit_post_pagination($wp_query, $this->get_id()); ?>
			</div>
<?php
		}
		wp_reset_postdata();
	}
}
