<?php

namespace UltimatePostKitPro\Modules\IconicSlider\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Background;
use Elementor\Plugin;

use UltimatePostKitPro\Utils;

use UltimatePostKit\Traits\Global_Widget_Controls;
use UltimatePostKit\Traits\Global_Widget_Functions;
use UltimatePostKit\Includes\Controls\GroupQuery\Group_Control_Query;
use UltimatePostKit\Modules\QueryControl\Controls\Group_Control_Posts;
use WP_Query;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Iconic_Slider extends Group_Control_Query {

	use Global_Widget_Controls;
		
	private $_query = null;

	public function get_name() {
		return 'upk-iconic-slider';
	}

	public function get_title() {
		return BDTUPK . esc_html__('Iconic Slider', 'ultimate-post-kit-pro');
	}

	public function get_icon() {
		return 'upk-widget-icon upk-icon-iconic-slider upk-new';
	}

	public function get_categories() {
		return ['ultimate-post-kit-pro'];
	}

	public function get_keywords() {
		return ['post', 'carousel', 'blog', 'recent', 'news', 'slider', 'iconic'];
	}

	public function get_style_depends() {
		if ($this->upk_is_edit_mode()) {
			return ['swiper', 'upk-all-styles-pro'];
		} else {
			return ['swiper', 'upk-font', 'upk-iconic-slider'];
		}
	}

	public function get_script_depends() {
		if ( $this->upk_is_edit_mode() ) {
			return [ 'swiper', 'upk-all-scripts-pro' ];
		} else {
			return [ 'swiper', 'upk-iconic-slider' ];
		}
	}

	public function get_custom_help_url() {
		return 'https://youtu.be/vsqsA_iwMU4?si=cnHyH-dOvGRgfvPX';
	}

	public function get_query() {
		return $this->_query;
	}

	public function has_widget_inner_wrapper(): bool {
        return ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
    }
	protected function is_dynamic_content(): bool {
		return true;
	}

	protected function register_controls() {
		$this->start_controls_section(
			'section_content_layout',
			[
				'label' => esc_html__('Layout', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_responsive_control(
			'item_height',
			[
				'label' => esc_html__('Height', 'ultimate-post-kit-pro'),
				'type'  => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'vh', 'em' ],
				'range' => [
					'px' => [
						'min' => 200,
						'max' => 1080,
					],
					'vh' => [
						'min' => 10,
						'max' => 100,
					],
				],
				'selectors'   => [
					'{{WRAPPER}} .upk-iconic-slider .upk-main-slider' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'content_alignment',
			[
				'label'     => esc_html__( 'Content Alignment', 'ultimate-post-kit-pro' ),
				'type'      => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options'   => [
					'left'   => [
						'title' => esc_html__( 'Left', 'ultimate-post-kit-pro' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'ultimate-post-kit-pro' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'ultimate-post-kit-pro' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-title, {{WRAPPER}} .upk-iconic-slider .upk-text' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'      => 'primary_thumbnail',
				'exclude'   => ['custom'],
				'default'   => 'full',
			]
		);

		$this->add_control(
			'hr_1',
			[
				'type'    => Controls_Manager::DIVIDER,
			]
		);

		//Global Title Controls
		$this->register_title_controls();

		$this->add_control(
			'show_category',
			[
				'label'   => esc_html__('Show Category', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);
		
		$this->add_control(
			'show_excerpt',
			[
				'label'   => esc_html__( 'Show Text', 'ultimate-post-kit-pro' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
			]
		);
		
		$this->add_control(
			'excerpt_length',
			[
				'label'       => esc_html__( 'Text Limit', 'ultimate-post-kit-pro' ),
				'description' => esc_html__( 'It\'s just work for main content, but not working with excerpt. If you set 0 so you will get full main content.', 'ultimate-post-kit-pro' ),
				'type'        => Controls_Manager::NUMBER,
				'default'     => 30,
				'condition'   => [
					'show_excerpt' => 'yes'
				],
			]
		);
		
		$this->add_control(
			'strip_shortcode',
			[
				'label'     => esc_html__( 'Strip Shortcode', 'ultimate-post-kit-pro' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'show_readmore',
			[
				'label' => esc_html__('Read More', 'ultimate-post-kit-pro'),
				'type'  => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'readmore_text',
			[
				'label'       => __( 'Readmore Text', 'ultimate-post-kit-pro' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__('Explore', 'ultimate-post-kit-pro'),
				'label_block' => false,
				'condition' => [
					'show_readmore' => 'yes'
				]
			]
		);

		//Global Date Controls
		$this->register_date_controls();

		//Global Reading Time Controls
		$this->register_reading_time_controls();

		$this->add_control(
			'show_pagination',
			[
				'label' => esc_html__( 'Show Pagination', 'ultimate-post-kit-pro' ),
				'type'  => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'show_navigation',
			[
				'label' => esc_html__( 'Show Arrows', 'ultimate-post-kit-pro' ),
				'type'  => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		// Query Settings
		$this->start_controls_section(
			'section_post_query_builder',
			[
				'label' => __( 'Query', 'ultimate-post-kit-pro' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'item_limit',
			[
				'label' => esc_html__('Item Limit', 'ultimate-post-kit-pro'),
				'type'  => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 1,
						'max' => 20,
					],
				],
				'default' => [
					'size' => 3,
				],
			]
		);
		
		$this->register_query_builder_controls();
		
		$this->end_controls_section();

		$this->start_controls_section(
			'section_carousel_settings',
			[
				'label' => __( 'Slider Settings', 'ultimate-post-kit-pro' ),
			]
		);

		$this->add_control(
			'autoplay',
			[
				'label'   => __( 'Autoplay', 'ultimate-post-kit-pro' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				
			]
		);

		$this->add_control(
			'autoplay_speed',
			[
				'label'     => esc_html__( 'Autoplay Speed', 'ultimate-post-kit-pro' ),
				'type'      => Controls_Manager::NUMBER,
				'default'   => 5000,
				'condition' => [
					'autoplay' => 'yes',
				],
			]
		);

		$this->add_control(
			'pauseonhover',
			[
				'label' => esc_html__( 'Pause on Hover', 'ultimate-post-kit-pro' ),
				'type'  => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
			'grab_cursor',
			[
				'label'   => __( 'Grab Cursor', 'ultimate-post-kit-pro' ),
				'type'    => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
			'loop',
			[
				'label'   => __( 'Loop', 'ultimate-post-kit-pro' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				
			]
		);


		$this->add_control(
			'speed',
			[
				'label'   => __( 'Animation Speed (ms)', 'ultimate-post-kit-pro' ),
				'type'    => Controls_Manager::SLIDER,
				'default' => [
					'size' => 800,
				],
				'range' => [
					'px' => [
						'min'  => 100,
						'max'  => 5000,
						'step' => 50,
					],
				],
			]
		);

		$this->add_control(
			'observer',
			[
				'label'       => __( 'Observer', 'ultimate-post-kit-pro' ),
				'description' => __( 'When you use carousel in any hidden place (in tabs, accordion etc) keep it yes.', 'ultimate-post-kit-pro' ),
				'type'        => Controls_Manager::SWITCHER,				
			]
		);

		$this->end_controls_section();

		//Style

		$this->start_controls_section(
			'section_style_slider_content',
			[
				'label' => esc_html__('Slider', 'ultimate-post-kit-pro'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'overlay_type',
			[
				'label'   => esc_html__('Overlay', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SELECT,
				'default' => 'background',
				'options' => [
					'none'       => esc_html__('None', 'ultimate-post-kit-pro'),
					'background' => esc_html__('Background', 'ultimate-post-kit-pro'),
					'blend'      => esc_html__('Blend', 'ultimate-post-kit-pro'),
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'overlay_color',
				'label' => esc_html__('Background', 'ultimate-post-kit-pro'),
				'types' => ['classic', 'gradient'],
				'exclude' => ['image'],
				'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-img-wrap::before',
				'fields_options' => [
					'background' => [
						'default' => 'gradient',
					],
					'color' => [
						'default' => 'rgba(3, 4, 16, 0)',
					],
					'color_b' => [
						'default' => 'rgba(3, 4, 16, 0.7)',
					],
				],
				'condition' => [
					'overlay_type' => ['background', 'blend'],
				],
			]
		);

		$this->add_control(
			'blend_type',
			[
				'label'     => esc_html__('Blend Type', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'multiply',
				'options'   => ultimate_post_kit_blend_options(),
				'condition' => [
					'overlay_type' => 'blend',
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-img-wrap::before' => 'mix-blend-mode: {{VALUE}};'
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_top_content',
			[
				'label' => esc_html__('Top Content', 'ultimate-post-kit-pro'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'top_content_padding',
			[
				'label' 	 => __('Padding', 'ultimate-post-kit-pro'),
				'type' 		 => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-top-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'top_content_margin',
			[
				'label' 	 => __('Margin', 'ultimate-post-kit-pro'),
				'type' 		 => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-top-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'content_width',
			[
				'label' => esc_html__('Width(%)', 'ultimate-post-kit-pro'),
				'type'  => Controls_Manager::SLIDER,
				'selectors'   => [
					'{{WRAPPER}} .upk-iconic-slider .upk-top-content' => 'width: {{SIZE}}%;',
				],
			]
		);

		$this->add_responsive_control(
			'content_min_width',
			[
				'label' => esc_html__('Min Width', 'ultimate-post-kit-pro'),
				'type'  => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'default' => [
					'unit' => 'px',
				],
				'range' => [
					'px' => [
						'min' => 200,
						'max' => 1200,
					],
					'%' => [
						'min' => 10,
						'max' => 100,
					],
				],
				'selectors'   => [
					'{{WRAPPER}} .upk-iconic-slider .upk-top-content' => 'min-width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs(
			'style_top_content_tabs'
		);
		
		$this->start_controls_tab(
			'style_title_tab',
			[
				'label' => esc_html__( 'Title', 'ultimate-post-kit-pro' ),
				'condition' => [
					'show_title' => 'yes',
				],
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_hover_color',
			[
				'label'     => esc_html__('Hover Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-title a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'title_margin',
			[
				'label' 	 => __('Margin', 'ultimate-post-kit-pro'),
				'type' 		 => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'      => 'title_typography',
				'label'     => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector'  => '{{WRAPPER}} .upk-iconic-slider .upk-title',
			]
		);

		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'title_text_shadow',
				'label' => __('Text Shadow', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-title a',
			]
		);

		
		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_date_tab',
			[
				'label' => esc_html__( 'Date', 'ultimate-post-kit-pro' ),
			]
		);

		$this->add_control(
			'date_color',
			[
				'label'     => esc_html__( 'Color', 'ultimate-post-kit-pro' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-date-and-time' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'date_background',
				'selector'  => '{{WRAPPER}} .upk-iconic-slider .upk-date-wrap::before',
			]
		);

		
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'        => 'date_border',
				'selector'    => '{{WRAPPER}} .upk-iconic-slider .upk-date-wrap::before',
			]
		);

		$this->add_responsive_control(
			'date_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-date-wrap::before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'date_typography',
				'label'    => esc_html__( 'Typography', 'ultimate-post-kit-pro' ),
				'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-date-and-time',
			]
		);
		
		$this->end_controls_tab();
		
		$this->end_controls_tabs();

		$this->end_controls_section();


		$this->start_controls_section(
			'section_style_bottom_content',
			[
				'label' => esc_html__('Bottom Content', 'ultimate-post-kit-pro'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'bottom_content_padding',
			[
				'label' 	 => __('Padding', 'ultimate-post-kit-pro'),
				'type' 		 => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-bottom-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'bottom_content_margin',
			[
				'label' 	 => __('Margin', 'ultimate-post-kit-pro'),
				'type' 		 => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-bottom-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs(
			'style_bottom_content_tabs'
		);
		
		$this->start_controls_tab(
			'style_text_tab',
			[
				'label' => esc_html__( 'Text', 'ultimate-post-kit-pro' ),
				'condition' => [
					'show_excerpt' => 'yes'
				]
			]
		);

		$this->add_control(
			'text_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'text_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-text',
			]
		);

		$this->add_responsive_control(
			'text_spacing',
			[
				'label'   => esc_html__('Spacing', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-desc-line-wrap' => 'grid-column-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'line_heading',
			[
				'label' => esc_html__( 'LINE', 'ultimate-post-kit-pro' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'line_background',
				'selector'  => '{{WRAPPER}} .upk-iconic-slider .upk-line',
			]
		);


		$this->add_responsive_control(
			'line_height',
			[
				'label'   => esc_html__('Height', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-line' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);
		// width
		$this->add_responsive_control(
			'line_width',
			[
				'label'   => esc_html__('Width', 'ultimate-post-kit-pro') . BDTUPK_NC,
				'type'    => Controls_Manager::SLIDER,
				'range' => [
					'%' => [
						'min'  => 0,
						'max'  => 100,
					],
					'px' => [
						'min'  => 0,
						'max'  => 1000,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-line' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_category_tab',
			[
				'label' => esc_html__( 'Category', 'ultimate-post-kit-pro' ),
				'condition' => [
					'show_category' => 'yes'
				],
			]
		);

		$this->add_control(
			'category_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-category a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'category_background',
				'selector'  => '{{WRAPPER}} .upk-iconic-slider .upk-category a',
			]
		);

		$this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'           => 'category_border',
                'label'          => __( 'Border', 'ultimate-post-kit-pro' ),
                'fields_options' => [
                    'border' => [
                        'default' => 'solid',
                    ],
                    'width'  => [
                        'default' => [
                            'top'      => '1',
                            'right'    => '1',
                            'bottom'   => '1',
                            'left'     => '1',
                            'isLinked' => false,
                        ],
                    ],
                    'color'  => [
                        'default' => '#fff',
                    ],
                ],
                'selector'       => '{{WRAPPER}} .upk-iconic-slider .upk-category a',
            ]
        );



		$this->add_responsive_control(
			'category_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-category a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-category a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_margin',
			[
				'label'      => esc_html__('Margin', 'ultimate-post-kit-pro') . BDTUPK_NC,
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-meta' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_spacing',
			[
				'label'   => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-category a+a' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'category_shadow',
				'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-category a',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'category_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-category a',
			]
		);


		$this->add_control(
			'category_heading',
			[
				'label' => esc_html__( 'Category Hover', 'ultimate-post-kit-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'category_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-category a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'category_hover_background',
				'selector'  => '{{WRAPPER}} .upk-iconic-slider .upk-category a:hover',
			]
		);

		$this->add_control(
			'category_hover_border_color',
			[
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'condition' => [
					'category_border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-category a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		
		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_reading_time_tab',
			[
				'label' => esc_html__( 'Reading Time', 'ultimate-post-kit-pro' ) . BDTUPK_NC,
				'condition' => [
					'show_reading_time' => 'yes'
				],
			]
		);

		$this->add_control(
			'reading_time_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-reading-time' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_reading_time' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'reading_time_margin',
			[
				'label'      => esc_html__('Margin', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-reading-time' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'show_reading_time' => 'yes'
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'reading_time_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-reading-time',
				'condition' => [
					'show_reading_time' => 'yes'
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_link_btn_tab',
			[
				'label' => esc_html__( 'Button', 'ultimate-post-kit-pro' ),
				'condition' => [
					'show_category' => 'yes'
				],
			]
		);

		$this->add_control(
			'link_btn_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-link-btn a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'link_btn_background',
				'selector'  => '{{WRAPPER}} .upk-iconic-slider .upk-link-btn a',
			]
		);

		$this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'           => 'link_btn_border',
                'label'          => __( 'Border', 'ultimate-post-kit-pro' ),
                'fields_options' => [
                    'border' => [
                        'default' => 'solid',
                    ],
                    'width'  => [
                        'default' => [
                            'top'      => '1',
                            'right'    => '1',
                            'bottom'   => '1',
                            'left'     => '1',
                            'isLinked' => false,
                        ],
                    ],
                    'color'  => [
                        'default' => '#fff',
                    ],
                ],
                'selector'       => '{{WRAPPER}} .upk-iconic-slider .upk-link-btn a',
            ]
        );



		$this->add_responsive_control(
			'link_btn_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-link-btn a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'link_btn_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-link-btn a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'link_btn_size',
			[
				'label'   => esc_html__('Size', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min'  => 0,
						'max'  => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-link-btn a' => 'height: {{SIZE}}{{UNIT}}; width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'link_btn_shadow',
				'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-link-btn a',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'link_btn_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-link-btn a',
			]
		);


		$this->add_control(
			'link_btn_heading',
			[
				'label' => esc_html__( 'Button Hover', 'ultimate-post-kit-pro' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'link_btn_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-link-btn a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'link_btn_hover_background',
				'selector'  => '{{WRAPPER}} .upk-iconic-slider .upk-link-btn a:hover',
			]
		);

		$this->add_control(
			'link_btn_hover_border_color',
			[
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'condition' => [
					'category_border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-link-btn a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();
		
		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_navigation',
			[
				'label'     => esc_html__('Navigation', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_navigation' => 'yes'
				],
			]
		);

		$this->add_control(
			'nav_arrows_icon',
			[
				'label'     => esc_html__('Arrows Icon', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SELECT,
				'default'   => '0',
				'options'   => [
					'0'        => esc_html__('Default', 'ultimate-post-kit-pro'),
					'1'        => esc_html__('Style 1', 'ultimate-post-kit-pro'),
					'2'        => esc_html__('Style 2', 'ultimate-post-kit-pro'),
					'3'        => esc_html__('Style 3', 'ultimate-post-kit-pro'),
					'4'        => esc_html__('Style 4', 'ultimate-post-kit-pro'),
					'5'        => esc_html__('Style 5', 'ultimate-post-kit-pro'),
					'6'        => esc_html__('Style 6', 'ultimate-post-kit-pro'),
					'7'        => esc_html__('Style 7', 'ultimate-post-kit-pro'),
					'8'        => esc_html__('Style 8', 'ultimate-post-kit-pro'),
					'9'        => esc_html__('Style 9', 'ultimate-post-kit-pro'),
					'10'       => esc_html__('Style 10', 'ultimate-post-kit-pro'),
					'11'       => esc_html__('Style 11', 'ultimate-post-kit-pro'),
					'12'       => esc_html__('Style 12', 'ultimate-post-kit-pro'),
					'13'       => esc_html__('Style 13', 'ultimate-post-kit-pro'),
					'14'       => esc_html__('Style 14', 'ultimate-post-kit-pro'),
					'15'       => esc_html__('Style 15', 'ultimate-post-kit-pro'),
					'16'       => esc_html__('Style 16', 'ultimate-post-kit-pro'),
					'17'       => esc_html__('Style 17', 'ultimate-post-kit-pro'),
					'18'       => esc_html__('Style 18', 'ultimate-post-kit-pro'),
					'circle-1' => esc_html__('Style 19', 'ultimate-post-kit-pro'),
					'circle-2' => esc_html__('Style 20', 'ultimate-post-kit-pro'),
					'circle-3' => esc_html__('Style 21', 'ultimate-post-kit-pro'),
					'circle-4' => esc_html__('Style 22', 'ultimate-post-kit-pro'),
					'square-1' => esc_html__('Style 23', 'ultimate-post-kit-pro'),
				],
			]
		);

		$this->start_controls_tabs('tabs_navigation_arrows_style');

		$this->start_controls_tab(
			'tabs_nav_arrows_normal',
			[
				'label'     => __('Normal', 'ultimate-post-kit-pro'),
				
			]
		);

		$this->add_control(
			'arrows_color',
			[
				'label'     => __('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-nav-button i' => 'color: {{VALUE}}',
				],
				
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'arrows_background',
				'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-nav-button',
			]
		);

		$this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'           => 'nav_arrows_border',
                'label'          => __( 'Border', 'ultimate-post-kit-pro' ),
                'fields_options' => [
                    'border' => [
                        'default' => 'solid',
                    ],
                    'width'  => [
                        'default' => [
                            'top'      => '1',
                            'right'    => '1',
                            'bottom'   => '1',
                            'left'     => '1',
                            'isLinked' => false,
                        ],
                    ],
                    'color'  => [
                        'default' => 'rgba(255, 255, 255, 0.7)',
                    ],
                ],
                'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-nav-button',
            ]
        );

		$this->add_responsive_control(
			'border_radius',
			[
				'label'      => __('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-nav-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				
			]
		);

		$this->add_responsive_control(
			'arrows_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-nav-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				
			]
		);

		$this->add_responsive_control(
			'arrows_margin',
			[
				'label'      => esc_html__('Margin', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-navigation-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				
			]
		);

		$this->add_responsive_control(
			'arrows_spacing',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-navigation-wrap' => 'grid-column-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'arrows_size',
			[
				'label'     => __('Size', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 10,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-nav-button i' => 'font-size: {{SIZE || 18}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tabs_nav_arrows_hover',
			[
				'label'     => __('Hover', 'ultimate-post-kit-pro'),
				
			]
		);

		$this->add_control(
			'arrows_hover_color',
			[
				'label'     => __('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-nav-button:hover i' => 'color: {{VALUE}}',
				],
				
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'arrows_hover_background',
				'selector' => '{{WRAPPER}} .upk-iconic-slider .upk-nav-button:hover',
			]
		);

		$this->add_control(
			'nav_arrows_hover_border_color',
			[
				'label'     => __('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .upk-nav-button:hover' => 'border-color: {{VALUE}};',
				],
				'condition' => [
					'nav_arrows_border_border!' => ''
				]
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		//Pagination Css
		$this->start_controls_section(
			'section_style_pagination',
			[
				'label'     => __( 'Pagination', 'ultimate-post-kit-pro' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'pagination_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .swiper-pagination-bullet' => 'color: {{VALUE}};',
				],
			]
		);
		//pagination active color
		$this->add_control(
			'pagination_active_color',
			[
				'label'     => esc_html__('Active Color', 'ultimate-post-kit-pro') . BDTUPK_NC,
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .swiper-pagination-bullet.swiper-pagination-bullet-active' => 'color: {{VALUE}};',
				],
			]
		);
		//line color
		$this->add_control(
			'pagination_line_color',
			[
				'label'     => esc_html__('Line Color', 'ultimate-post-kit-pro') . BDTUPK_NC,
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .swiper-pagination-bullet::after' => 'background-color: {{VALUE}};',
				],
			]
		);

		//line width
		$this->add_responsive_control(
			'pagination_line_width',
			[
				'label'     => esc_html__('Line Width', 'ultimate-post-kit-pro') . BDTUPK_NC,
				'type'      => Controls_Manager::SLIDER,
				'size_units'=> ['px'],
				'range'     => [
					'px' => [
						'min' => 20,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-iconic-slider .swiper-pagination-bullet.swiper-pagination-bullet-active::after' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'pagination_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-iconic-slider .swiper-pagination-bullet',
			]
		);

		$this->add_responsive_control(
			'nav_pag_content_margin',
			[
				'label' 	 => __('Margin', 'ultimate-post-kit-pro'),
				'type' 		 => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-iconic-slider .upk-pagination' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		
		$this->end_controls_section();
			
	}

	/**
	 * Main query render for this widget
	 * @param $posts_per_page number item query limit
	 */
	public function query_posts( $posts_per_page ) {
		
		$default = $this->getGroupControlQueryArgs();
		if ( $posts_per_page ) {
			$args['posts_per_page'] = $posts_per_page;
				$args['paged']  = max( 1, get_query_var( 'paged' ), get_query_var( 'page' ) );
		}
		$args         = array_merge( $default, $args );
		$this->_query = new WP_Query( $args );
	}

	public function render_image($image_id, $size) {
		$placeholder_image_src = Utils::get_placeholder_image_src();

		$image_src = wp_get_attachment_image_src($image_id, $size);

		if (!$image_src) {
			$image_src = $placeholder_image_src;
		} else {
			$image_src = $image_src[0];
		}

		?>

		<img class="upk-img" src="<?php echo esc_url($image_src); ?>" alt="<?php echo esc_attr(get_the_title()); ?>">

		<?php
	}

	public function render_title() {
		$settings = $this->get_settings_for_display();

		if (!$this->get_settings('show_title')) {
			return;
		}

		$this->add_render_attribute('slider-title', 'class', 'upk-title', true);
		$titleClass = $this->get_render_attribute_string('slider-title');
		echo
		'<' . esc_html($settings['title_tags']) . ' ' . wp_kses_post($titleClass) . ' >
				<a href="' . esc_url(get_permalink()) . '" title="' . esc_attr(get_the_title()) . '">
					' . esc_html(get_the_title())  . '
				</a>
			</' . esc_html($settings['title_tags']) . '>';

	}

	public function render_category() {

		if (!$this->get_settings('show_category')) {
			return;
		}
		?>
		<div class="upk-category">
			<?php echo upk_get_category($this->get_settings('posts_source')); ?>
		</div>
		<?php
	}

	public function render_date() {
		$settings = $this->get_settings_for_display();
		
		
		if ( ! $this->get_settings( 'show_date' ) ) {
			return;
		}
		
		?>
		<div class="upk-date-and-time upk-flex upk-flex-middle">
			<div class="upk-date-wrap">
				<span><?php if ( $settings['human_diff_time'] == 'yes' ) {
						echo ultimate_post_kit_post_time_diff( ( $settings['human_diff_time_short'] == 'yes' ) ? 'short' : '' );
					} else {
						echo get_the_date();
					} ?>
				</span>

			</div>
			<?php if ($settings['show_time']) : ?>
			<div class="upk-post-time">
				<i class="upk-icon-clock" aria-hidden="true"></i>
				<?php echo get_the_time(); ?>
			</div>
			<?php endif; ?>
		</div>
		
		<?php
	}

	public function render_excerpt( $excerpt_length ) {
			
		if ( ! $this->get_settings( 'show_excerpt' ) ) {
			return;
		}
		$strip_shortcode = $this->get_settings_for_display( 'strip_shortcode' );
		?>
		<div class="upk-text">
			<?php
				if ( has_excerpt() ) {
					the_excerpt();
				} else {
					echo ultimate_post_kit_custom_excerpt( $excerpt_length, $strip_shortcode );
				}
			?>
		</div>
		
		<?php
	}
	
	public function render_header() {
		$id              = 'upk-iconic-slider-' . $this->get_id();
		$settings        = $this->get_settings_for_display();
		
		$this->add_render_attribute( 'iconic-slider', 'id', $id );
		$this->add_render_attribute( 'iconic-slider', 'class', ['upk-iconic-slider'] );

		$this->add_render_attribute(
			[
				'iconic-slider' => [
					'data-settings' => [
						wp_json_encode(array_filter([
							"autoplay"       => ( "yes" == $settings["autoplay"] ) ? [ "delay" => $settings["autoplay_speed"] ] : false,
							"loop"           => ($settings["loop"] == "yes") ? true : false,
							"speed"          => $settings["speed"]["size"],
							"effect"         => 'fade',
							"lazy"           => true,
							"parallax"       => true,
							"grabCursor"     => ($settings["grab_cursor"] === "yes") ? true : false,
							"pauseOnHover"   => ("yes" == $settings["pauseonhover"]) ? true : false,
							"slidesPerView"  => 1,
							"observer"       => ($settings["observer"]) ? true : false,
							"observeParents" => ($settings["observer"]) ? true : false,
			      	        "navigation" => [
			      				"nextEl" => "#" . $id . " .upk-navigation-next",
			      				"prevEl" => "#" . $id . " .upk-navigation-prev",
			      			],
			      			"pagination" => [
								"el"             => "#" . $id . " .upk-pagination",
								"clickable"      => "true",
							],
				        ]))
					]
				]
			]
		);

		$this->add_render_attribute('swiper', 'class', 'upk-main-slider swiper');

		?>
		<div <?php echo $this->get_render_attribute_string( 'iconic-slider' ); ?>>
			<div <?php echo $this->get_render_attribute_string( 'swiper' ); ?>>
				<div class="swiper-wrapper">
		<?php
	}

	public function render_footer() {
		$settings = $this->get_settings_for_display();
		
		?>
			</div>

			<div class="upk-navi-pag-wrap">
				<?php if($settings['show_pagination']) : ?>
					    <div class="upk-pagination"></div>
				<?php endif; ?>

				<?php if($settings['show_navigation']) : ?>
					<div class="upk-navigation-wrap">
						<div class="upk-nav-button upk-navigation-prev">
							<i class="upk-icon-arrow-right-<?php echo esc_attr($settings['nav_arrows_icon']); ?>" aria-hidden="true"></i>
						</div>
						<div class="upk-nav-button upk-navigation-next">
							<i class="upk-icon-arrow-left-<?php echo esc_attr($settings['nav_arrows_icon']); ?>" aria-hidden="true"></i>
						</div>
					</div>
				<?php endif; ?>

		    	</div>
			</div>
		</div>

		<?php
	}

	public function render_post_grid_item($post_id, $image_size, $excerpt_length) {
		$settings = $this->get_settings_for_display();

		$this->add_render_attribute('slider-item', 'class', 'upk-item swiper-slide', true);

		?>

		<div <?php echo $this->get_render_attribute_string('slider-item'); ?> data-swiper-parallax-opacity="0">
		    <div class="upk-img-wrap" data-swiper-parallax="-100" data-swiper-parallax-scale="1.2">
				<?php $this->render_image(get_post_thumbnail_id($post_id), $image_size); ?>
			</div>

			<div class="upk-top-content">
			    <?php if ($settings['show_date'] ) : ?>
			       <?php $this->render_date(); ?>
				<?php endif; ?>
				<div data-swiper-parallax-y="-50" data-swiper-parallax-scale=".8">
					<?php $this->render_title(substr($this->get_name(), 4)); ?>
				</div>
			</div>

			<div class="upk-bottom-content">

			   <?php if ($settings['show_readmore'] === 'yes' and !empty($settings['readmore_text'])) : ?>
					<div class="upk-link-btn">
						<a href="<?php echo esc_url(get_permalink()); ?>">
							<span><?php echo esc_html($settings['readmore_text']); ?></span>
							<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-right" viewBox="0 0 16 16">
							<path fill-rule="evenodd" d="M1 8a.5.5 0 0 1 .5-.5h11.793l-3.147-3.146a.5.5 0 0 1 .708-.708l4 4a.5.5 0 0 1 0 .708l-4 4a.5.5 0 0 1-.708-.708L13.293 8.5H1.5A.5.5 0 0 1 1 8z"/>
							</svg>
						</a>
					</div>
				<?php endif; ?>

				<div class="bdt-category-text-wrap">
					<div class="upk-meta upk-flex upk-flex-middle" data-swiper-parallax="200">
						<?php $this->render_category(); ?>
						<?php if ('yes' === $settings['show_reading_time']) : ?>
							<div class="upk-reading-time">
								<?php echo ultimate_post_kit_reading_time(get_the_content(), $settings['avg_reading_speed']); ?>
							</div>
						<?php endif; ?>
					</div>
	
					<div class="upk-desc-line-wrap" data-swiper-parallax="200">
						<?php if($settings['show_excerpt']) : ?>
							<?php $this->render_excerpt( $excerpt_length ); ?>
						<?php endif; ?>
						<div class="upk-line"></div>
					</div>
				</div>

            </div>
		</div>

		<?php
	}

	public function render() {
		$settings = $this->get_settings_for_display();

		$this->query_posts($settings['item_limit']['size']);
		$wp_query = $this->get_query();

		if (!$wp_query->found_posts) {
			return;
		}
		
		$this->render_header();

		while ( $wp_query->have_posts() ) {
			$wp_query->the_post();
			$thumbnail_size = $settings['primary_thumbnail_size'];

			$this->render_post_grid_item( get_the_ID(), $thumbnail_size, $settings['excerpt_length'] );
		}

		$this->render_footer();

		wp_reset_postdata();
	}
}
