<?php

namespace UltimatePostKitPro\Modules\HoluxTabs\Widgets;

use UltimatePostKitPro\Base\Module_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Background;
use Elementor\Repeater;

use UltimatePostKitPro\Utils;
use UltimatePostKit\Traits\Global_Widget_Controls;


if (!defined('ABSPATH')) {
	exit;
} // Exit if accessed directly

class Holux_Tabs extends Module_Base {

	use Global_Widget_Controls;

	public function get_name() {
		return 'upk-holux-tabs';
	}

	public function get_title() {
		return BDTUPK . esc_html__('Holux Tabs', 'ultimate-post-kit-pro');
	}

	public function get_icon() {
		return 'upk-widget-icon upk-icon-holux-tabs';
	}

	public function get_categories() {
		return ['ultimate-post-kit-pro'];
	}

	public function get_keywords() {
		return ['post', 'blog', 'recent', 'news', 'holux', 'list', 'tabs'];
	}

	public function get_style_depends() {
		if ($this->upk_is_edit_mode()) {
			return ['upk-all-styles-pro'];
		} else {
			return ['upk-font', 'upk-holux-tabs'];
		}
	}

	public function get_script_depends() {
		if ($this->upk_is_edit_mode()) {
			return ['fslightbox', 'upk-all-scripts-pro'];
		} else {
			return ['fslightbox', 'upk-holux-tabs'];
		}
	}

	public function get_custom_help_url() {
		return 'https://youtu.be/P-y7v3RRP1M';
	}

	public function has_widget_inner_wrapper(): bool {
        return ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
    }
	protected function is_dynamic_content(): bool {
		return true;
	}

	protected function register_controls() {
		$this->start_controls_section(
			'section_post_grid_query',
			[
				'label' => esc_html__('Query', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'item_limit',
			[
				'label'   => esc_html__('Item Limit', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SLIDER,
				'range'   => [
					'px' => [
						'min' => 1,
						'max' => 20,
					],
				],
				'default' => [
					'size' => 6,
				],
			]
		);
		$this->add_control(
			'order',
			[
				'label'   => esc_html__('Order', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SELECT,
				'default' => 'desc',
				'options' => [
					'asc'  => esc_html__('ASC', 'ultimate-post-kit-pro'),
					'desc' => esc_html__('DESC', 'ultimate-post-kit-pro'),
				],
			]
		);
		$repeater = new Repeater();
		$repeater->add_control(
			'filter_label',
			[
				'label'       => esc_html__('Label', 'ultimate-post-kit-pro'),
				'type'        => Controls_Manager::TEXT,
			]
		);
		$repeater->add_control(
			'custom_post_type',
			[
				'label'       => esc_html__('Post Type', 'ultimate-post-kit-pro'),
				'type'        => Controls_Manager::SELECT,
				'default'     => 'post',
				'options'     => ultimate_post_kit_get_post_types(),
			]
		);
		$repeater->add_control(
			'post_format',
			[
				'label'      => esc_html__('Post Format', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::SELECT,
				'default'    => 'all',
				'options'    => [
					'all' 		=> esc_html__('All', 'ultimate-post-kit-pro'),
					'standard' 	=> esc_html__('Standard', 'ultimate-post-kit-pro'),
					'aside' 	=> esc_html__('Aside', 'ultimate-post-kit-pro'),
					'chat' 		=> esc_html__('Chat', 'ultimate-post-kit-pro'),
					'gallery' 	=> esc_html__('Gallery ', 'ultimate-post-kit-pro'),
					'link' 		=> esc_html__('Link', 'ultimate-post-kit-pro'),
					'image' 	=> esc_html__('Image', 'ultimate-post-kit-pro'),
					'quote' 	=> esc_html__('Quote', 'ultimate-post-kit-pro'),
					'status' 	=> esc_html__('Status', 'ultimate-post-kit-pro'),
					'audio' 	=> esc_html__('Audio', 'ultimate-post-kit-pro'),
					'video' 	=> esc_html__('Video', 'ultimate-post-kit-pro'),
				],
				'condition' => [
					'custom_post_type' => 'post'
				]
			]
		);
		$repeater->add_control(
			'filter_by',
			[
				'label'      => esc_html__('Filter By', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::SELECT,
				'default'    => 'latest',
				'options'    => [
					'latest' => esc_html__('Latest', 'ultimate-post-kit-pro'),
					'trending' => esc_html__('Trending', 'ultimate-post-kit-pro'),
					'popular' => esc_html__('Popular', 'ultimate-post-kit-pro'),
					'modified' => esc_html__('Last Modified', 'ultimate-post-kit-pro'),
					'video' => esc_html__('Video', 'ultimate-post-kit-pro'),
					'random' => esc_html__('Random', 'ultimate-post-kit-pro'),
				]
			]
		);

		$repeater->add_control(
			'trending_days_limit',
			[
				'label'     => esc_html__('Trending Days Limit', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::NUMBER,
				'default'   => 20,
				'condition' => [
					'filter_by' =>  'trending'
				]
			]
		);

		$this->add_control(
			'filter_item_list',
			[
				'label'         => esc_html__('Filter List', 'ultimate-post-kit-pro'),
				'type'          => Controls_Manager::REPEATER,
				'fields'        => $repeater->get_controls(),
				'default'       => [
					[
						'filter_label'    => esc_html__('Latest', 'ultimate-post-kit-pro'),
						'filter_by'    => esc_html__('latest', 'ultimate-post-kit-pro'),
					],
					[
						'filter_label'    => esc_html__('Trending', 'ultimate-post-kit-pro'),
						'filter_by'      =>   esc_html__('trending', 'ultimate-post-kit-pro')
					],
					[
						'filter_label'    => esc_html__('Popular', 'ultimate-post-kit-pro'),
						'filter_by' => esc_html__('popular', 'ultimate-post-kit-pro')
					],
				],
				'title_field'   => '{{{ filter_label }}}',
			]
		);


		$this->end_controls_section();

		$this->start_controls_section(
			'section_addition',
			[
				'label' => esc_html__('Additional Options', 'ultimate-post-kit-pro'),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);
		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'      => 'primary_thumbnail',
				'exclude'   => ['custom'],
				'default'   => 'medium',
			]
		);
		$this->add_control(
			'show_title',
			[
				'label'         => esc_html__('Show Title', 'ultimate-post-kit-pro'),
				'type'          => Controls_Manager::SWITCHER,
				'label_on'      => esc_html__('Show', 'ultimate-post-kit-pro'),
				'label_off'     => esc_html__('Hide', 'ultimate-post-kit-pro'),
				'return_value'  => 'yes',
				'default'       => 'yes',
				'separator'     => 'before',
			]
		);
		$this->add_control(
			'title_tags',
			[
				'label'     => __('Title HTML Tag', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'h3',
				'options'   => ultimate_post_kit_title_tags(),
				'condition' => [
					'show_title' => 'yes'
				]
			]
		);
		$this->add_control(
			'show_category',
			[
				'label'         => esc_html__('Show Category', 'ultimate-post-kit-pro'),
				'type'          => Controls_Manager::SWITCHER,
				'label_on'      => esc_html__('Show', 'ultimate-post-kit-pro'),
				'label_off'     => esc_html__('Hide', 'ultimate-post-kit-pro'),
				'return_value'  => 'yes',
				'default'       => 'yes',
				'separator'     => 'before',
			]
		);

		$this->add_control(
			'show_meta',
			[
				'label'         => esc_html__('Show Meta', 'ultimate-post-kit-pro'),
				'type'          => Controls_Manager::SWITCHER,
				'label_on'      => esc_html__('Show', 'ultimate-post-kit-pro'),
				'label_off'     => esc_html__('Hide', 'ultimate-post-kit-pro'),
				'return_value'  => 'yes',
				'default'       => 'yes',
			]
		);
		$this->add_control(
			'show_author',
			[
				'label'         => esc_html__('Show Author', 'ultimate-post-kit-pro'),
				'type'          => Controls_Manager::SWITCHER,
				'label_on'      => esc_html__('Show', 'ultimate-post-kit-pro'),
				'label_off'     => esc_html__('Hide', 'ultimate-post-kit-pro'),
				'return_value'  => 'yes',
				'default'       => 'yes',
				'condition' => [
					'show_meta' => 'yes'
				]
			]
		);
		$this->add_control(
			'show_date',
			[
				'label'         => esc_html__('Show Date', 'ultimate-post-kit-pro'),
				'type'          => Controls_Manager::SWITCHER,
				'label_on'      => esc_html__('Show', 'ultimate-post-kit-pro'),
				'label_off'     => esc_html__('Hide', 'ultimate-post-kit-pro'),
				'return_value'  => 'yes',
				'default'       => 'yes',
				'condition' => [
					'show_meta' => 'yes'
				]
			]
		);
		$this->add_control(
			'meta_separator',
			[
				'label'       => __('Separator', 'ultimate-post-kit-pro'),
				'type'        => Controls_Manager::TEXT,
				'default'     => '//',
				'label_block' => false,
			]
		);

		//Global Reading Time Controls
		$this->register_reading_time_controls();
		
		$this->end_controls_section();
		$this->start_controls_section(
			'section_style_tabs',
			[
				'label' => esc_html__('Tabs', 'ultimate-post-kit-pro'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->start_controls_tabs(
			'style_tabs'
		);
		$this->start_controls_tab(
			'tab_normal',
			[
				'label' => esc_html__('Normal', 'ultimate-post-kit-pro'),
			]
		);
		$this->add_control(
			'tab_normal_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs-header .upk-list a' => 'color: {{VALUE}}',
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'tab_normal_background',
				'label'     => esc_html__('Background', 'ultimate-post-kit-pro'),
				'types'     => ['classic', 'gradient'],
				'selector'  => '{{WRAPPER}} .upk-holux-tabs-header .upk-list a',
			]
		);
		
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'tabs_border',
				'label'     => esc_html__('Border', 'ultimate-post-kit-pro'),
				'selector'  => '{{WRAPPER}} .upk-holux-tabs-header li',
				'separator' => 'before'
			]
		);
		$this->add_responsive_control(
			'tabs_border_radius',
			[
				'label'                 => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'                  => Controls_Manager::DIMENSIONS,
				'size_units'            => ['px', '%', 'em'],
				'selectors'             => [
					'{{WRAPPER}} .upk-holux-tabs-header li'    => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};overflow: hidden;',
				],
			]
		);
		$this->add_responsive_control(
			'tabs_height',
			[
				'label'         => esc_html__('Height', 'ultimate-post-kit-pro'),
				'type'          => Controls_Manager::SLIDER,
				'size_units'    => ['px'],
				'range'         => [
					'px'        => [
						'min'   => 20,
						'max'   => 200,
						'step'  => 1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs-header a' => 'height: {{SIZE}}{{UNIT}}; line-height: {{SIZE}}{{UNIT}};',
				],
				'separator' => 'before'
			]
		);
		$this->add_responsive_control(
			'tabs_spacing',
			[
				'label'         => esc_html__('Spacing', 'ultimate-post-kit-pro'),
				'type'          => Controls_Manager::SLIDER,
				'size_units'    => ['px'],
				'range'         => [
					'px'        => [
						'min'   => 0,
						'max'   => 30,
						'step'  => 1,
					],
				],
				'selectors'             => [
					'{{WRAPPER}} .upk-holux-tabs-header li + li'    => 'margin-left: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'      => 'tabs_typography',
				'label'     => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'exclude' => ['line_height', 'text_decoration'],
				'selector'  => '{{WRAPPER}} .upk-holux-tabs-header a',
			]
		);
		$this->end_controls_tab();
		$this->start_controls_tab(
			'tab_active',
			[
				'label' => esc_html__('Active', 'ultimate-post-kit-pro'),
			]
		);
		$this->add_control(
			'tab_active_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs-header .upk-active a' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'tab_normal_border_color',
			[
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs-header .upk-list.upk-active' => 'border-color: {{VALUE}}',
				],
				'condition' => [
					'tabs_border_border!' => ''
				]
			]
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'tab_active_background',
				'label'     => esc_html__('Background', 'ultimate-post-kit-pro'),
				'types'     => ['classic', 'gradient'],
				'selector'  => '{{WRAPPER}} .upk-holux-tabs-header .upk-active a',
			]
		);
		
		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();
		$this->start_controls_section(
			'upk_section_style',
			[
				'label' => esc_html__('Items', 'ultimate-post-kit-pro'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'items_background',
				'label'     => esc_html__('Background', 'ultimate-post-kit-pro'),
				'types'     => ['classic', 'gradient'],
				'selector'  => '{{WRAPPER}} .upk-holux-tabs .upk-item',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'items_border',
				'label'     => esc_html__('Border', 'ultimate-post-kit-pro'),
				'selector'  => '{{WRAPPER}} .upk-holux-tabs .upk-item',
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'items_border_radius',
			[
				'label'                 => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'                  => Controls_Manager::DIMENSIONS,
				'size_units'            => ['px', '%', 'em'],
				'selectors'             => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item'    => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};overflow: hidden;',
				],
			]
		);

		$this->add_responsive_control(
			'items_padding',
			[
				'label'                 => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'                  => Controls_Manager::DIMENSIONS,
				'size_units'            => ['px', '%', 'em'],
				'selectors'             => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item'    => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'items_gap',
			[
				'label'                 => esc_html__('Item Gap', 'ultimate-post-kit-pro'),
				'type'                  => Controls_Manager::SLIDER,
				'size_units'            => ['px'],
				'range'                 => [
					'px'    => [
						'min'   => 0,
						'max'   => 100,
					],
				],
				'selectors'             => [
					'{{WRAPPER}} .upk-holux-tabs'    => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'items_content_padding',
			[
				'label'                 => esc_html__('Content Padding', 'ultimate-post-kit-pro'),
				'type'                  => Controls_Manager::DIMENSIONS,
				'size_units'            => ['px', '%', 'em'],
				'selectors'             => [
					'{{WRAPPER}} .upk-item .upk-content'    => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator'             => 'before'
			]
		);

		$this->add_responsive_control(
			'wrapper_padding',
			[
				'label' => __('Wrap Margin', 'ultimate-post-kit-pro'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'default'   => [
					'top' => 30,
					'right' => 0,
					'bottom' => 0,
					'left' => 0,
					'isLinked' => true,
				],
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_image',
			[
				'label'     => esc_html__('Image', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_responsive_control(
			'items_image_size',
			[
				'label'         => esc_html__('Size', 'ultimate-post-kit-pro'),
				'type'          => Controls_Manager::SLIDER,
				'size_units'    => ['%', 'px', 'vh'],
				'range'         => [
					'px'        => [
						'min'   => 50,
						'max'   => 300,
						'step'  => 1,
					],
					'%'         => [
						'min'   => 1,
						'max'   => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-item .upk-image' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'item_image_border',
				'selector' => '{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-image a img',
			]
		);

		$this->add_responsive_control(
			'item_image_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-image a img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'      => 'item_image_box_shadow',
				'label'     => esc_html__('Box Shadow', 'ultimate-post-kit-pro'),
				'selector'  => '{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-image a img',
			]
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_title',
			[
				'label'     => esc_html__('Title', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_hover_color',
			[
				'label'     => esc_html__('Hover Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-title a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'title_spacing',
			[
				'label'     => esc_html__('Bottom Spacing', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-title a',
			]
		);

		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name'     => 'title_text_shadow',
				'label'    => __('Text Shadow', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-title a',
			]
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_meta',
			[
				'label'     => esc_html__('Meta', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'meta_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}}  .upk-holux-tabs .upk-content .upk-meta *' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'meta_hover_color',
			[
				'label'     => esc_html__('Author Hover Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-meta a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'meta_spacing',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-item .upk-meta' => 'gap: 0 {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'meta_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-meta *',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_category',
			[
				'label'     => esc_html__('Category', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);
		$this->start_controls_tabs('tabs_category_style');
		$this->start_controls_tab(
			'tab_category_normal',
			[
				'label' => esc_html__('Normal', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'category_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-category a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'category_background',
			[
				'label'     => esc_html__('Background Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-category a' => 'background: {{VALUE}}',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'category_border',
				'label'     => esc_html__('Border', 'ultimate-post-kit-pro'),
				'selector'  => '{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-category a',
			]
		);
		$this->add_responsive_control(
			'category_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-category a ' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-category a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'category_margin',
			[
				'label'      => esc_html__('Margin', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-category' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_spacing',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-item .upk-category' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'      => 'category_box_shadow',
				'label'     => esc_html__('Box Shadow', 'ultimate-post-kit-pro'),
				'selector'  => '{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-category a',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'category_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'exclude' => ['line_height'],
				'selector' => '{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-category a ',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_category_hover',
			[
				'label' => esc_html__('Hover', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'category_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-category a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'category_hover_bg',
			[
				'label'     => esc_html__('Background Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-category a:hover' => 'background: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'category_hover_border_color',
			[
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'condition' => [
					'category_border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-item .upk-item-box .upk-content .upk-category a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_lightbox',
			[
				'label'     => esc_html__('Lightbox Icon', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->start_controls_tabs('tabs_lightbox_style');
		$this->start_controls_tab(
			'tab_lightbox_normal',
			[
				'label' => esc_html__('Normal', 'ultimate-post-kit-pro'),
			]
		);
		$this->add_control(
			'lightbox_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-play-button' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'lightbox_shadow_pulse_color',
			[
				'label'     => esc_html__('Shadow Pulse Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-play-button:before, {{WRAPPER}} .upk-holux-tabs .upk-play-button:after' => 'border-color: {{VALUE}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'lightbox_background',
				'types'     => ['classic', 'gradient'],
				'exclude'    => ['image'],
				'selector'  => '{{WRAPPER}} .upk-holux-tabs .upk-play-button',
			]
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'lightbox_border',
				'selector'  => '{{WRAPPER}} .upk-holux-tabs .upk-play-button',
				'separator' => 'before'
			]
		);
		$this->add_responsive_control(
			'lightbox_border_radius',
			[
				'label'                 => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'                  => Controls_Manager::DIMENSIONS,
				'size_units'            => ['px', '%', 'em'],
				'selectors'             => [
					'{{WRAPPER}} .upk-holux-tabs .upk-play-button, {{WRAPPER}} .upk-holux-tabs .upk-play-button:before, {{WRAPPER}} .upk-holux-tabs .upk-play-button:after'    => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'lightbox_padding',
			[
				'label'                 => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'                  => Controls_Manager::DIMENSIONS,
				'size_units'            => ['px', '%', 'em'],
				'selectors'             => [
					'{{WRAPPER}} .upk-holux-tabs .upk-play-button'    => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'lightbox_font_size',
			[
				'label'                 => esc_html__('Font Size', 'ultimate-post-kit-pro'),
				'type'                  => Controls_Manager::SLIDER,
				'size_units'            => ['px'],
				'range'                 => [
					'px'    => [
						'min'   => 10,
						'max'   => 50,
					],
				],
				'selectors'             => [
					'{{WRAPPER}} .upk-holux-tabs .upk-play-button'    => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_tab();
		$this->start_controls_tab(
			'tab_lightbox_hover',
			[
				'label' => esc_html__('Hover', 'ultimate-post-kit-pro'),
			]
		);
		$this->add_control(
			'lightbox_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-play-button:hover' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'lightbox_hover_background',
				'types'     => ['classic', 'gradient'],
				'exclude'    => ['image'],
				'selector'  => '{{WRAPPER}} .upk-holux-tabs .upk-play-button:hover',
			]
		);
		$this->add_control(
			'lightbox_hover_border_color',
			[
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'condition' => [
					'lightbox_border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .upk-holux-tabs .upk-play-button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function render_title() {
		$settings = $this->get_settings_for_display();
		if (!$this->get_settings('show_title')) {
			return;
		}

		printf('<%1$s class="upk-title"><a href="%2$s" title="%3$s" class="upk-title">%3$s</a></%1$s>', esc_attr(Utils::get_valid_html_tag($settings['title_tags'])), get_permalink(), get_the_title());

	}


	public function render_author() { ?>
		<div class="upk-author">
			<a href="<?php echo get_author_posts_url(get_the_author_meta('ID')) ?>">
				<i class="upk-icon-user"></i>
				<span class="upk-author-name"><?php echo get_the_author() ?></span>
			</a>
		</div>
	<?php
	}

	public function render_date() { ?>
		<div class="upk-date">
			<i class="upk-icon upk-icon-calendar"></i>
			<span class="upk-date-text">
				<?php echo get_the_date(); ?>
			</span>
		</div>
	<?php
	}

	public function render_category() {
		$settings = $this->get_settings_for_display();
		foreach ($settings['filter_item_list'] as $key => $value) {
			if ($key == 0) :
				$post_type = $value['custom_post_type'];
			endif;
		}

		?>
		<div class="upk-category">
			<?php echo upk_get_category($post_type); ?>
		</div>
		<?php
	}
	protected function render_header() {
		$settings = $this->get_settings_for_display();
		foreach ($settings['filter_item_list'] as $key => $value) {
			$trending_days_limit = $value['trending_days_limit'];
		}
		$this->add_render_attribute('holux-tabs', 'class', 'ultimate-post-kit-holux-tabs-wrap', true);
		$this->add_render_attribute(
			[
				'holux-tabs' => [
					'data-settings' => [
						wp_json_encode(array_filter(
							[
								'order' 				=> $settings['order'],
								'posts_per_page'   		=>  $settings['item_limit']['size'],
								'trending_days_limit' 	=> $trending_days_limit,
								'show_title' 			=>  isset($settings['show_title']) ? $settings['show_title'] : 'yes',
								'title_tags' 			=>  isset($settings['title_tags']) ? $settings['title_tags'] : 'h3',
								'show_category' 		=>  isset($settings['show_category']) ? $settings['show_category'] : 'yes',
								'show_meta' 			=>  isset($settings['show_meta']) ? $settings['show_meta'] : 'yes',
								'show_author' 			=>  isset($settings['show_author']) ? $settings['show_author'] : 'yes',
								'show_date' 			=>  isset($settings['show_date']) ? $settings['show_date'] : 'yes',
								'show_reading_time' 	=>  isset($settings['show_reading_time']) ? $settings['show_reading_time'] : 'yes',
								// 'avg_reading_speed' 		=>  $settings['avg_reading_speed'],
								'meta_separator' 		=> $settings['meta_separator'],
							]
						))
					]
				]
			]
		);

	?>
		<div <?php $this->print_render_attribute_string('holux-tabs'); ?>>
			<ul class="upk-holux-tabs-header">
				<?php
				if ($settings['filter_item_list']) {
					foreach ($settings['filter_item_list'] as $key => $item) {
						$this->add_render_attribute(
							[
								'holux-tab-option' => [
									'data-settings' => [
										wp_json_encode(array_filter([
											'post_type'   =>  $item['custom_post_type'],
											'post_format' => $item['post_format'],
											'filter_by'   => $item['filter_by'],
										]))
									]
								]
							],
							'',
							'',
							true
						);
						$active_class = ($key == 0) ? 'upk-active' : '';
				?>
						<li class="upk-list <?php echo esc_attr($active_class); ?>">
							<?php printf('<a href="javascript:void(0)" class="post-tab-option" %2$s>%1$s</a>', esc_html($item['filter_label']), $this->get_render_attribute_string('holux-tab-option')); ?>
						</li>
				<?php
					}
				}
				?>
			</ul>
			<div class="upk-holux-tabs">
			<?php
		}
		protected function render_footer() { ?>
			</div>
		</div>
		<?php
		}
		protected function render_loop_item() {
			$settings = $this->get_settings_for_display();
			if (!(empty($settings['filter_item_list']))) {
				foreach ($settings['filter_item_list'] as $key => $tag) {
					if ($key == 0) :
						$post_type   =  $tag['custom_post_type'];
						$post_format = $tag['post_format'];
						$filter_by  = $tag['filter_by'];
						$trending_days_limit  = $tag['trending_days_limit'];
					endif;
				}
			}
			
			$query_args = [
				'post_type' => $post_type,
				'post_status' => 'publish',
				'posts_per_page' => $settings['item_limit']['size'],
				'order' => $settings['order'],
				'ignore_sticky_posts' => 1
			];

			if ($post_format == 'post') :
				switch ($post_format) {
					case 'standard':
						$query_args['tax_query'] = [
							[
								'taxonomy' => 'post_format',
								'field' => 'slug',
								'terms' => [
									'post-format-aside', 'post-format-audio', 'post-format-chat', 'post-format-gallery', 'post-format-link', 'post-format-image', 'post-format-quote', 'post-format-status', 'post-format-video',
								],
								'operator' => 'NOT IN'
							]
						];
						break;
					case 'all':
						$query_args['tax_query'] = [];
						break;
					default:
						$query_args['tax_query'] = [
							[
								'taxonomy' => 'post_format',
								'field' => 'slug',
								'terms' => ['post-format-' . $post_format . ''],
							]
						];
						break;
				}
			endif;


			switch ($filter_by) {

				case 'popular':
					$query_args['orderby'] = 'comment_count';
					$query_args['order'] = 'DESC';
					break;
				case 'trending':
					$query_args['orderby'] = 'comment_count';
					$query_args['oroder'] = 'DESC';
					$query_args['date_query'] = [
						'after'  => '' . $trending_days_limit . ' days ago',
						'inclusive' => true,
					];
					break;
				case 'random':
					$query_args['orderby'] = 'rand';
					$query_args['order'] = 'DESC';
					break;
				case 'modified':
					$query_args['orderby'] = 'modified';
					$query_args['order'] = 'DESC';
					break;

				case 'video':
					$query_args['meta_query'] = [
						[
							'key' => '_upk_video_link_meta_key',
							'compare' => 'EXISTS'
						]
					];
					$query_args['order'] = 'DESC';
					break;
				default:
					$query_args['orderby'] = 'date';
					$query_args['order'] = 'DESC';
					break;
			}

			$wp_query = new \WP_Query($query_args);
			if (!$wp_query->found_posts) {
				return;
			}


			while ($wp_query->have_posts()) :
				$wp_query->the_post();

				$video_link = get_post_meta(get_the_ID(), '_upk_video_link_meta_key', true);
				if ($video_link !== false) {
					$video_link = $this->video_link_render($video_link);
				}

				if ($filter_by === 'video' && $video_link !== false) {
					$this->render_tabs_content( $filter_by, $video_link ); 

				} elseif ($filter_by !== 'video') {
					$this->render_tabs_content( $filter_by, false );
				} ?>
				<?php endwhile;
			wp_reset_postdata();
		}

		public function render_tabs_content( $filter_by, $video_link ) {
			$settings = $this->get_settings_for_display();

			$id = get_the_ID();

			$image_src = wp_get_attachment_image_url(get_post_thumbnail_id(), $settings['primary_thumbnail_size']);
			$placeholder_image_src = Utils::get_placeholder_image_src();
			if (empty($image_src)) {
				$image_src = $placeholder_image_src;
			} 

			?>
			<div class="upk-item">
				<div class="upk-item-box">
					<div class="upk-image">
						<a href="<?php echo get_permalink(); ?>">
							<img class="upk-img" src="<?php echo esc_url($image_src) ?>" alt="<?php echo get_the_title(); ?>">
						</a>

						<?php if ( $filter_by === 'video' && $video_link !== false) : ?>
							<div class="upk-play-btn-wrap">
								<a data-fslightbox="lightbox" href="#lightbox-<?php echo esc_attr($id); ?>" class="upk-play-button">
									<i class="upk-icon-play" aria-hidden="true"></i>
								</a>
								<div class="usk-hidden">
									<iframe src="<?php echo esc_url($video_link); ?>" id="lightbox-<?php echo esc_attr($id); ?>" width="1920px" height="1080px" frameborder="0" allow="autoplay; fullscreen" allowfullscreen></iframe>
								</div>
							</div>
						<?php endif; ?>
					</div>
					<div class="upk-content">
						<?php
						if ($settings['show_category'] === 'yes') :
							$this->render_category();
						endif; ?>
						<?php
						if ($settings['show_title'] === 'yes') :
							$this->render_title(substr($this->get_name(), 4));
						endif;
						?>
						<?php //if ($settings['show_meta'] === 'yes') :
						?>
						<div class="upk-meta">
							<?php $this->render_author(); ?>

							<?php if ($settings['show_date']) : ?>
								<span class="upk-separator"><?php echo esc_html($settings['meta_separator']); ?></span>
								<?php $this->render_date(); ?>
							<?php endif; ?>

							<?php if ('yes' === $settings['show_reading_time']) : ?>
								<span class="upk-separator"><?php echo esc_html($settings['meta_separator']); ?></span>
								<div class="upk-reading-time">
									<?php echo ultimate_post_kit_reading_time(get_the_content(), $settings['avg_reading_speed']); ?>
								</div>
							<?php endif; ?>
						</div>
						
					</div>
				</div>
			</div>
			<?php
		}

		public function video_link_render($video) {
			$youtube_id = (preg_match('%(?:youtube(?:-nocookie)?\.com/(?:[^/]+/.+/|(?:v|e(?:mbed)?)/|.*[?&]v=)|youtu\.be/)([^"&?/ ]{11})%i', $video, $match)) ? $match[1] : false;
	
			$vimeo_id = (preg_match('%^https?:\/\/(?:www\.|player\.)?vimeo.com\/(?:channels\/(?:\w+\/)?|groups\/([^\/]*)\/videos\/|album\/(\d+)\/video\/|video\/|)(\d+)(?:$|\/|\?)(?:[?]?.*)$%im', $video, $match)) ? $match[3] : false;
	
			if ($youtube_id) {
				// $video_source    = 'https://www.youtube.com/watch?v=' . $youtube_id;
				$video_source = 'https://www.youtube.com/embed/' . $youtube_id;
			} elseif ($vimeo_id) {
				$video_source = 'https://vimeo.com/' . $vimeo_id;
			} else {
				$video_source = false;
			}
			return $video_source;
		}
		protected function render() {
			$this->render_header();
			$this->render_loop_item();
			$this->render_footer();
		}
	}
