<?php

namespace UltimatePostKitPro\Modules\EldoraCarousel\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Plugin;

use UltimatePostKitPro\Utils;

use UltimatePostKit\Traits\Global_Widget_Controls;
use UltimatePostKit\Traits\Global_Widget_Functions;
use UltimatePostKit\Traits\Global_Swiper_Functions;
use UltimatePostKit\Includes\Controls\GroupQuery\Group_Control_Query;
use WP_Query;

if (!defined('ABSPATH')) {
	exit;
} // Exit if accessed directly

class Eldora_Carousel extends Group_Control_Query {

	use Global_Widget_Controls;
	use Global_Widget_Functions;
	use Global_Swiper_Functions;

	private $_query = null;

	public function get_name() {
		return 'upk-eldora-carousel';
	}

	public function get_title() {
		return BDTUPK . esc_html__('Eldora Carousel', 'ultimate-post-kit-pro');
	}

	public function get_icon() {
		return 'upk-widget-icon upk-icon-eldora-carousel';
	}

	public function get_categories() {
		return ['ultimate-post-kit-pro'];
	}

	public function get_keywords() {
		return ['post', 'carousel', 'blog', 'recent', 'news', 'eldora'];
	}

	public function get_style_depends() {
		if ($this->upk_is_edit_mode()) {
			return ['swiper', 'upk-all-styles-pro'];
		} else {
			return ['swiper', 'upk-font', 'upk-eldora-carousel'];
		}
	}

	public function get_script_depends() {
		if ($this->upk_is_edit_mode()) {
			return ['swiper', 'upk-all-scripts-pro'];
		} else {
			return ['swiper', 'upk-eldora-carousel'];
		}
	}

	public function get_custom_help_url() {
		return 'https://youtu.be/oJXRzQpowJo';
	}

	public function get_query() {
		return $this->_query;
	}

	public function has_widget_inner_wrapper(): bool {
        return ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
    }
	protected function is_dynamic_content(): bool {
		return true;
	}

	protected function register_controls() {
		$this->start_controls_section(
			'section_content_layout',
			[
				'label' => esc_html__('Layout', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_responsive_control(
			'columns',
			[
				'label' => __('Columns', 'ultimate-post-kit-pro'),
				'type' => Controls_Manager::SELECT,
				'default' => 2,
				'tablet_default' => 2,
				'mobile_default' => 1,
				'options' => [
					1 => '1',
					2 => '2',
					3 => '3',
					4 => '4',
					5 => '5',
					6 => '6',
				],
			]
		);

		$this->add_responsive_control(
			'item_gap',
			[
				'label' => __('Item Gap', 'ultimate-post-kit-pro'),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 20,
				],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
			]
		);

		// item height control
		$this->add_responsive_control(
			'item_height',
			[
				'label' => __('Item Height', 'ultimate-post-kit-pro'),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 400,
				],
				'range' => [
					'px' => [
						'min' => 100,
						'max' => 1000,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-eldora-img-wrap' => 'height: {{SIZE}}{{UNIT}};',
				],
			],
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'    => 'primary_thumbnail',
				'exclude' => ['custom'],
				'default' => 'medium',
			]
		);

		$this->end_controls_section();

		//New Query Builder Settings
		$this->start_controls_section(
			'section_post_query_builder',
			[
				'label' => __('Query', 'ultimate-post-kit-pro'),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'item_limit',
			[
				'label'     => esc_html__('Item Limit', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 1,
						'max'  => 20,
					],
				],
				'default'   => [
					'size' => 6,
				],
			]
		);

		$this->register_query_builder_controls();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_content_additional',
			[
				'label' => esc_html__('Additional', 'ultimate-post-kit-pro'),
			]
		);

		//Global Title Controls
		$this->register_title_controls();

		$this->add_control(
			'show_category',
			[
				'label'   => esc_html__('Show Category', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		$this->add_control(
			'show_excerpt',
			[
				'label'   => esc_html__('Show Text', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
				
			]
		);

		$this->add_control(
			'excerpt_length',
			[
				'label'       => esc_html__('Text Limit', 'ultimate-post-kit-pro'),
				'description' => esc_html__('It\'s just work for main content, but not working with excerpt. If you set 0 so you will get full main content.', 'ultimate-post-kit-pro'),
				'type'        => Controls_Manager::NUMBER,
				'default'     => 15,
				'condition'   => [
					'show_excerpt' => 'yes'
				],
			]
		);

		$this->add_control(
			'strip_shortcode',
			[
				'label'     => esc_html__('Strip Shortcode', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'show_meta',
			[
				'label'     => esc_html__('Show Author', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'show_readmore',
			[
				'label' => esc_html__('Read more', 'ultimate-post-kit-pro'),
				'type'  => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'readmore_text',
			[
				'label'       => __('Readmore Text', 'ultimate-post-kit-pro'),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__('Explore', 'ultimate-post-kit-pro'),
				'label_block' => false,
				'condition' => [
					'show_readmore' => 'yes'
				]
			]
		);
		//Global Date Controls
		$this->register_date_controls();

		//Global Reading Time Controls
		$this->register_reading_time_controls();

		$this->add_control(
			'meta_separator',
			[
				'label'       => __('Separator', 'ultimate-post-kit-pro') . BDTUPK_NC,
				'type'        => Controls_Manager::TEXT,
				'default'     => '/',
				'label_block' => false,
			]
		);

		$this->add_control(
			'item_match_height',
			[
				'label'        => __('Item Match Height', 'ultimate-post-kit-pro'),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'yes',
				'prefix_class' => 'upk-item-match-height--',
				'separator'    => 'before'
			]
		);

		$this->add_control(
			'global_link',
			[
				'label'        => __('Item Wrapper Link', 'ultimate-post-kit-pro'),
				'type'         => Controls_Manager::SWITCHER,
				'prefix_class' => 'upk-global-link-',
				'description'  => __('Be aware! When Item Wrapper Link activated then title link and read more link will not work', 'ultimate-post-kit-pro'),
			]
		);


		$this->end_controls_section();

		//Navigaiton Global Controls
		$this->register_navigation_controls('eldora');

		//Style
		$this->start_controls_section(
			'upk_section_style',
			[
				'label' => esc_html__('Items', 'ultimate-post-kit-pro'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->update_control(
			'centered_slides',
			[
				'label'       => __('Center Slide', 'ultimate-post-kit-pro'),
				'description' => __('Use even items from Layout > Columns settings for better preview.', 'ultimate-post-kit-pro'),
				'type'        => Controls_Manager::HIDDEN,
				'default'     => 'yes',
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'item_background',
				'selector'  => '{{WRAPPER}} .upk-eldora-carousel .upk-eldora-content-wrap',
			]
		);
		
		$this->add_responsive_control(
			'item_content_padding',
			[
				'label'      => esc_html__('Content Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-eldora-inner-content, {{WRAPPER}} .upk-eldora-carousel .upk-eldora-link-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		// border
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'item_content_border',
				'selector'  => '{{WRAPPER}} .upk-eldora-carousel .upk-eldora-item',
			]
		);

		$this->add_responsive_control(
			'item_content_border_radius',
			[
				'label'      => esc_html__('Content Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-eldora-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);


		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_title',
			[
				'label'     => esc_html__('Title', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_title' => 'yes',
				],
			]
		);

		$this->add_control(
			'title_style',
			[
				'label'   => esc_html__('Style', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SELECT,
				'default' => 'underline',
				'options' => [
					'underline'        => esc_html__('Underline', 'ultimate-post-kit-pro'),
					'middle-underline' => esc_html__('Middle Underline', 'ultimate-post-kit-pro'),
					'overline'         => esc_html__('Overline', 'ultimate-post-kit-pro'),
					'middle-overline'  => esc_html__('Middle Overline', 'ultimate-post-kit-pro'),
				],
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_hover_color',
			[
				'label'     => esc_html__('Hover Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-title a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name'     => 'title_text_shadow',
				'label'    => __('Text Shadow', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-eldora-carousel .upk-title',
			]
		);

		$this->add_responsive_control(
			'title_spacing',
			[
				'label'     => esc_html__('Spacing', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-eldora-title .upk-title',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_text',
			[
				'label'     => esc_html__('Text', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'text_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-eldora-expert' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'text_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-eldora-carousel .upk-eldora-expert',
			]
		);

		$this->add_responsive_control(
			'text_margin',
			[
				'label'      => __('Margin', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-eldora-expert' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_meta',
			[
				'label'      => esc_html__('Author', 'ultimate-post-kit-pro'),
				'tab'        => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_meta' => 'yes',
				],
			]
		);

		$this->add_control(
			'meta_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-eldora-meta' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_hover_color',
			[
				'label'     => esc_html__('Hover Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-eldora-author-name:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'meta_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-eldora-carousel .upk-eldora-meta',
			]
		);

		$this->add_responsive_control(
			'meta_icon_size',
			[
				'label'     => esc_html__('Icon Size', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-eldora-meta svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);
		

		$this->add_responsive_control(
			'meta_space_between',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-eldora-author' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();


		$this->start_controls_section(
			'section_style_category',
			[
				'label'     => esc_html__('Category', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_category' => 'yes',
				],
			]
		);

		$this->start_controls_tabs('tabs_category_style');

		$this->start_controls_tab(
			'tab_category_normal',
			[
				'label' => esc_html__('Normal', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'category_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-category a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'category_background',
				'selector' => '{{WRAPPER}} .upk-eldora-category a',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'category_border',
				'selector' => '{{WRAPPER}} .upk-eldora-category a',
			]
		);

		$this->add_responsive_control(
			'category_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-eldora-category a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-eldora-category a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_spacing',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
						'step' => 2,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-category a+a' => 'margin-left: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'category_shadow',
				'selector' => '{{WRAPPER}} .upk-eldora-category a',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'category_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-eldora-category a',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_category_hover',
			[
				'label' => esc_html__('Hover', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'category_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-category a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'category_hover_background',
				'selector' => '{{WRAPPER}} .upk-eldora-category a:hover',
			]
		);

		$this->add_control(
			'category_hover_border_color',
			[
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'condition' => [
					'category_border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} {{WRAPPER}} .upk-eldora-category a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();


		$this->start_controls_section(
			'section_style_date',
			[
				'label'      => esc_html__('Date/Time', 'ultimate-post-kit-pro'),
				'tab'        => Controls_Manager::TAB_STYLE,
				'conditions' => [
					'relation' => 'or',
					'terms'    => [
						[
							'name'  => 'show_date',
							'value' => 'yes'
						],
						[
							'name'  => 'show_reading_time',
							'value' => 'yes'
						]
					]
				],
			]
		);

		$this->add_control(
			'date_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-reading-time, {{WRAPPER}} .upk-eldora-carousel .upk-post-time, {{WRAPPER}} .upk-eldora-carousel .upk-eldora-date' => 'color: {{VALUE}};',
				],
			]
		);


		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'date_background',
				'selector' => '.upk-eldora-carousel .upk-reading-time, {{WRAPPER}} .upk-eldora-carousel .upk-post-time, {{WRAPPER}} .upk-eldora-carousel .upk-eldora-date',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'date_border',
				'selector' => '.upk-eldora-carousel .upk-reading-time, {{WRAPPER}} .upk-eldora-carousel .upk-post-time, {{WRAPPER}} .upk-eldora-carousel .upk-eldora-date',
			]
		);

		$this->add_responsive_control(
			'date_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-reading-time, {{WRAPPER}} .upk-eldora-carousel .upk-post-time, {{WRAPPER}} .upk-eldora-carousel .upk-eldora-date' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);


		$this->add_responsive_control(
			'date_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-reading-time, {{WRAPPER}} .upk-eldora-carousel .upk-post-time, {{WRAPPER}} .upk-eldora-carousel .upk-eldora-date' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'date_space_between',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-carousel .upk-date-reading > div:before' => 'margin: 0 {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'show_reading_time' => 'yes',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'date_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-eldora-carousel .upk-reading-time, {{WRAPPER}} .upk-eldora-carousel .upk-post-time, {{WRAPPER}} .upk-eldora-carousel .upk-eldora-date',
			]
		);



		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'date_shadow',
				'selector' => '{{WRAPPER}} .upk-eldora-carousel .upk-reading-time, {{WRAPPER}} .upk-eldora-carousel .upk-post-time, {{WRAPPER}} .upk-eldora-carousel .upk-eldora-date',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_link_button',
			[
				'label'     => esc_html__('Read More', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_readmore' => 'yes',
				],
			]
		);

		$this->start_controls_tabs('tabs_link_button_style');

		$this->start_controls_tab(
			'tab_link_button_normal',
			[
				'label' => esc_html__('Normal', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'link_button_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-link-button a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'link_button_background',
				'selector' => '{{WRAPPER}} .upk-eldora-link-button a',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'link_button_border',
				'selector' => '{{WRAPPER}} .upk-eldora-link-button a',
			]
		);

		$this->add_responsive_control(
			'link_button_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-eldora-link-button a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'link_button_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-eldora-link-button a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'link_button_spacing',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
						'step' => 2,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-link-button a' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'link_button_shadow',
				'selector' => '{{WRAPPER}} .upk-eldora-link-button a',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'link_button_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-eldora-link-button a',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_link_button_hover',
			[
				'label' => esc_html__('Hover', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'link_button_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-eldora-link-button a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'link_button_hover_background',
				'selector' => '{{WRAPPER}} .upk-eldora-link-button a:before',
			]
		);

		$this->add_control(
			'link_button_hover_border_color',
			[
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'condition' => [
					'link_button_border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} {{WRAPPER}} .upk-eldora-link-button a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		//Navigation Global Controls
		$this->register_navigation_style('swiper');
	}

	/**
	 * Main query render for this widget
	 * @param $posts_per_page number item query limit
	 */
	public function query_posts($posts_per_page) {

		$default = $this->getGroupControlQueryArgs();
		if ($posts_per_page) {
			$args['posts_per_page'] = $posts_per_page;
			$args['paged']  = max(1, get_query_var('paged'), get_query_var('page'));
		}
		$args         = array_merge($default, $args);
		$this->_query = new WP_Query($args);
	}

	public function render_image($image_id, $size) {
		$placeholder_image_src = Utils::get_placeholder_image_src();

		$image_src = wp_get_attachment_image_src($image_id, $size);

		if (!$image_src) {
			$image_src = $placeholder_image_src;
		} else {
			$image_src = $image_src[0];
		}

?>

		<img class="upk-eldora-img" src="<?php echo esc_url($image_src); ?>" alt="<?php echo esc_html(get_the_title()); ?>">

	<?php
	}

	public function render_category() {

		if (!$this->get_settings('show_category')) {
			return;
		}
	?>
		<div class="upk-eldora-category">
			<?php echo upk_get_category($this->get_settings('posts_source')); ?>
		</div>
	<?php
	}

	public function render_date() {
		$settings = $this->get_settings_for_display();


		if (!$this->get_settings('show_date')) {
			return;
		}

	?>
			<div class="upk-eldora-date">
				<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-calendar" viewBox="0 0 16 16">
				<path d="M3.5 0a.5.5 0 0 1 .5.5V1h8V.5a.5.5 0 0 1 1 0V1h1a2 2 0 0 1 2 2v11a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2V3a2 2 0 0 1 2-2h1V.5a.5.5 0 0 1 .5-.5zM1 4v10a1 1 0 0 0 1 1h12a1 1 0 0 0 1-1V4H1z"/>
				</svg>
				<?php if ($settings['human_diff_time'] == 'yes') {
					echo ultimate_post_kit_post_time_diff(($settings['human_diff_time_short'] == 'yes') ? 'short' : '');
				} else {
					echo get_the_date();
				} ?>
			</div>
			<?php if ($settings['show_time']) : ?>
				<div class="upk-post-time">
					<i class="upk-icon-clock" aria-hidden="true"></i>
					<?php echo get_the_time(); ?>
				</div>
			<?php endif; ?>

	<?php
	}

	public function render_author() {

		if (!$this->get_settings('show_meta')) {
			return;
		}
	?>

		<div class="upk-eldora-author">
			<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-person-circle" viewBox="0 0 16 16">
				<path d="M11 6a3 3 0 1 1-6 0 3 3 0 0 1 6 0z"/>
				<path fill-rule="evenodd" d="M0 8a8 8 0 1 1 16 0A8 8 0 0 1 0 8zm8-7a7 7 0 0 0-5.468 11.37C3.242 11.226 4.805 10 8 10s4.757 1.225 5.468 2.37A7 7 0 0 0 8 1z"/>
			</svg>
			<span>by</span>
			<a class="upk-eldora-author-name" href="<?php echo get_author_posts_url(get_the_author_meta('ID')) ?>">
		    	<span><?php echo get_the_author() ?></span>
		    </a>
		</div>


	<?php
	}

	public function render_excerpt($excerpt_length) {

		if (!$this->get_settings('show_excerpt')) {
			return;
		}
		$strip_shortcode = $this->get_settings_for_display('strip_shortcode');
	?>
		<div class="upk-eldora-expert" data-swiper-parallax-X="-100" data-swiper-parallax-duration="800">
			<?php
			if (has_excerpt()) {
				the_excerpt();
			} else {
				echo ultimate_post_kit_custom_excerpt($excerpt_length, $strip_shortcode);
			}
			?>
		</div>

	<?php
	}

	public function render_header_attribute( $name ) {
		$id              = 'upk-' . $name . '-carousel-' . $this->get_id();
		$settings        = $this->get_settings_for_display();
		$elementor_vp_lg = get_option( 'elementor_viewport_lg' );
		$elementor_vp_md = get_option( 'elementor_viewport_md' );
		$viewport_lg     = ! empty( $elementor_vp_lg ) ? $elementor_vp_lg - 1 : 1023;
		$viewport_md     = ! empty( $elementor_vp_md ) ? $elementor_vp_md - 1 : 767;
		
		$this->add_render_attribute( 'carousel', 'id', $id );
		$this->add_render_attribute( 'carousel', 'class', [ 'upk-' . $name . '-carousel' ] );
		
		if ( 'arrows' == $settings['navigation'] ) {
			$this->add_render_attribute( 'carousel', 'class', 'upk-arrows-align-' . $settings['arrows_position'] );
		} elseif ( 'dots' == $settings['navigation'] ) {
			$this->add_render_attribute( 'carousel', 'class', 'upk-dots-align-' . $settings['dots_position'] );
		} elseif ( 'both' == $settings['navigation'] ) {
			$this->add_render_attribute( 'carousel', 'class', 'upk-arrows-dots-align-' . $settings['both_position'] );
		} elseif ( 'arrows-fraction' == $settings['navigation'] ) {
			$this->add_render_attribute( 'carousel', 'class', 'upk-arrows-dots-align-' . $settings['arrows_fraction_position'] );
		}
		
		if ( 'arrows-fraction' == $settings['navigation'] ) {
			$pagination_type = 'fraction';
		} elseif ( 'both' == $settings['navigation'] or 'dots' == $settings['navigation'] ) {
			$pagination_type = 'bullets';
		} elseif ( 'progressbar' == $settings['navigation'] ) {
			$pagination_type = 'progressbar';
		} else {
			$pagination_type = '';
		}
		
		$this->add_render_attribute(
			[
				'carousel' => [
					'data-settings' => [
						wp_json_encode( array_filter( [
							"autoplay"              => ( "yes" == $settings["autoplay"] ) ? [ "delay" => $settings["autoplay_speed"] ] : false,
							"loop"                  => ( $settings["loop"] == "yes" ) ? true : false,
							"speed"                 => $settings["speed"]["size"],
							"pauseOnHover"          => ( "yes" == $settings["pauseonhover"] ) ? true : false,
							"slidesPerView"         => isset($settings["columns_mobile"]) ? (int)$settings["columns_mobile"] : 1,
							"slidesPerGroup"        => isset($settings["slides_to_scroll_mobile"]) ? (int)$settings["slides_to_scroll_mobile"] : 1,
							"spaceBetween"          => !empty($settings["item_gap_mobile"]["size"]) ? (int)$settings["item_gap_mobile"]["size"] : 20,
							"centeredSlides"        => ( $settings["centered_slides"] === "yes" ) ? true : false,
							"grabCursor"            => ( $settings["grab_cursor"] === "yes" ) ? true : false,
							"effect"                => $settings["skin"],
							"observer"              => ( $settings["observer"] ) ? true : false,
							"observeParents"        => ( $settings["observer"] ) ? true : false,
							"direction"             => $settings['direction'],
							"watchSlidesVisibility" => true,
							"watchSlidesProgress"   => true,
							"parallax"   => true,
							"breakpoints"           => [
								(int) $viewport_md => [
									"slidesPerView"  => isset($settings["columns_tablet"]) ? (int)$settings["columns_tablet"] : 2,
									"spaceBetween"   => !empty($settings["item_gap_tablet"]["size"]) ? (int)$settings["item_gap_tablet"]["size"] : 20,
									"slidesPerGroup" => isset($settings["slides_to_scroll_tablet"]) ? (int)$settings["slides_to_scroll_tablet"] : 1,
								],
								(int) $viewport_lg => [
									"slidesPerView"  => isset($settings["columns"]) ? (int)$settings["columns"] : 3,
									"spaceBetween"   => !empty($settings["item_gap"]["size"]) ? (int)$settings["item_gap"]["size"] : 20,
									"slidesPerGroup" => isset($settings["slides_to_scroll"]) ? (int)$settings["slides_to_scroll"] : 1,
								]
							],
							"navigation"            => [
								"nextEl" => "#" . $id . " .upk-navigation-next",
								"prevEl" => "#" . $id . " .upk-navigation-prev",
							],
							"pagination"            => [
								"el"             => "#" . $id . " .swiper-pagination",
								"type"           => $pagination_type,
								"clickable"      => "true",
								'dynamicBullets' => ( "yes" == $settings["dynamic_bullets"] ) ? true : false,
							],
							"scrollbar"             => [
								"el"   => "#" . $id . " .swiper-scrollbar",
								"hide" => "true",
							],
							'coverflowEffect'       => [
								'rotate'       => ( "yes" == $settings["coverflow_toggle"] ) ? $settings["coverflow_rotate"]["size"] : 50,
								'stretch'      => ( "yes" == $settings["coverflow_toggle"] ) ? $settings["coverflow_stretch"]["size"] : 0,
								'depth'        => ( "yes" == $settings["coverflow_toggle"] ) ? $settings["coverflow_depth"]["size"] : 100,
								'modifier'     => ( "yes" == $settings["coverflow_toggle"] ) ? $settings["coverflow_modifier"]["size"] : 1,
								'slideShadows' => true,
							],
						
						] ) )
					]
				]
			]
		);

		$this->add_render_attribute('swiper', 'class', 'swiper-carousel swiper');
	}

	public function render_header() {
		//Global Function
		$this->render_header_attribute('eldora');

	?>
		<div <?php $this->print_render_attribute_string('carousel'); ?>>


			<div class="upk-eldora-carousel-wrapper">

				<div <?php $this->print_render_attribute_string('swiper'); ?>>
					<div class="swiper-wrapper">
					<?php
				}
				
				
				public function render_post_carousel_item($post_id, $image_size, $excerpt_length) {
					$settings = $this->get_settings_for_display();

					if ('yes' == $settings['global_link']) {

						$this->add_render_attribute('carousel-item', 'onclick', "window.open('" . esc_url(get_permalink()) . "', '_self')", true);
					}

					$this->add_render_attribute('carousel-item', 'class', 'upk-eldora-item swiper-slide upk-transition-toggle', true);

					?>
						<div <?php $this->print_render_attribute_string('carousel-item'); ?>>

							<div class="upk-eldora-img-wrap">
								 <div class="upk-category-view-wrap">
									<?php $this->render_category(); ?>
								 </div>
								<?php $this->render_image(get_post_thumbnail_id($post_id), $image_size); ?>

								<?php if ($settings['show_date'] or $settings['show_reading_time']) : ?>
									<div class="upk-date-reading upk-flex upk-flex-middle">
										<?php $this->render_date(); ?>
										<?php if ('yes' === $settings['show_reading_time']) : ?>
											<div class="upk-reading-time" data-separator="<?php echo esc_html($settings['meta_separator']); ?>">
												<?php echo ultimate_post_kit_reading_time(get_the_content(), $settings['avg_reading_speed']); ?>
											</div>
										<?php endif; ?>
									</div>
								<?php endif; ?>

							</div>

							<div class="upk-eldora-content-wrap">
								<div class="upk-eldora-inner-content">

							    	<div class="upk-eldora-title " data-swiper-parallax-X="-200" data-swiper-parallax-duration="800" >
										<?php $this->render_title(substr($this->get_name(), 4)); ?>
									</div>
									<?php if ($settings['show_meta']) : ?>
										<div class="upk-eldora-meta" data-swiper-parallax-X="-150" data-swiper-parallax-duration="800">
											<?php $this->render_author(); ?>
										</div>
									<?php endif; ?>

									<?php $this->render_excerpt($excerpt_length); ?>
								</div>

								<?php if ($settings['show_readmore'] === 'yes' and !empty($settings['readmore_text'])) : ?>
									<div class="upk-eldora-link-button" data-swiper-parallax-X="-150" data-swiper-parallax-duration="600">
										<a href="<?php echo esc_url(get_permalink()); ?>">
											<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-right" viewBox="0 0 16 16">
												<path fill-rule="evenodd" d="M1 8a.5.5 0 0 1 .5-.5h11.793l-3.147-3.146a.5.5 0 0 1 .708-.708l4 4a.5.5 0 0 1 0 .708l-4 4a.5.5 0 0 1-.708-.708L13.293 8.5H1.5A.5.5 0 0 1 1 8z"/>
											</svg>
											<span><?php echo esc_html($settings['readmore_text']); ?></span>
										</a>
									</div>
								<?php endif; ?>

							</div>

						</div>
				<?php
				}

				public function render() {
					$settings = $this->get_settings_for_display();
	 
					$this->query_posts($settings['item_limit']['size']);
					$wp_query = $this->get_query();

					if (!$wp_query->found_posts) {
						return;
					}

					$this->render_header();

					while ($wp_query->have_posts()) {
						$wp_query->the_post();
						$thumbnail_size = $settings['primary_thumbnail_size'];

						$this->render_post_carousel_item(get_the_ID(), $thumbnail_size, $settings['excerpt_length']);
					}

					$this->render_footer();

					wp_reset_postdata();
				}
			}
