<?php

namespace UltimatePostKitPro\Modules\Checkerboard\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;

use UltimatePostKit\Traits\Global_Widget_Controls;
use UltimatePostKit\Traits\Global_Widget_Functions;
use UltimatePostKit\Includes\Controls\GroupQuery\Group_Control_Query;
use WP_Query;

if (!defined('ABSPATH')) {
	exit;
} // Exit if accessed directly

class Checkerboard extends Group_Control_Query {

	use Global_Widget_Controls;
	use Global_Widget_Functions;

	private $_query = null;

	public function get_name() {
		return 'upk-checkerboard';
	}

	public function get_title() {
		return BDTUPK . esc_html__('Checkerboard', 'ultimate-post-kit-pro');
	}

	public function get_icon() {
		return 'upk-widget-icon upk-icon-checkerboard';
	}

	public function get_categories() {
		return ['ultimate-post-kit-pro'];
	}

	public function get_keywords() {
		return ['post', 'grid', 'blog', 'recent', 'news', 'checkerboard'];
	}

	public function get_style_depends() {
		if ($this->upk_is_edit_mode()) {
			return ['upk-all-styles-pro'];
		} else {
			return ['upk-font', 'upk-checkerboard'];
		}
	}

	public function get_custom_help_url() {
		return 'https://youtu.be/nqB_bGNPSts?si=KTWQOARPuL5lTdjM';
	}

	public function get_query() {
		return $this->_query;
	}

	public function has_widget_inner_wrapper(): bool {
        return ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
    }
	protected function is_dynamic_content(): bool {
		return true;
	}

	protected function register_controls() {
		$this->start_controls_section(
			'section_content_layout',
			[
				'label' => esc_html__('Layout', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'preset_style',
			[
				'label'   => esc_html__('Preset Style', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SELECT,
				'default' => 'style-1',
				'options' => [
					'style-1'  => esc_html__('Preset 1', 'ultimate-post-kit-pro'),
					'style-2'  => esc_html__('Preset 2', 'ultimate-post-kit-pro'),
				],
			]
		);

		$this->add_responsive_control(
			'row_gap',
			[
				'label'     => esc_html__('Row Gap', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'default'   => [
					'size' => 20,
				],
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'    => 'primary_thumbnail',
				'exclude' => ['custom'],
				'default' => 'medium',
			]
		);

		$this->add_responsive_control(
			'odd_content_text_align',
			[
				'label'     => esc_html__('Text Align', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => [
					'left'   => [
						'title' => esc_html__('Left', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__('Right', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-text-align-right',
					],
					'justify'  => [
						'title' => esc_html__('Justify', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-text-align-justify',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-content' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'odd_meta_text_align',
			[
				'label'     => esc_html__('Meta Align', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => [
					'left'   => [
						'title' => esc_html__('Left', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__('Right', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-text-align-right',
					],
					'space-between'  => [
						'title' => esc_html__('Justify', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-text-align-justify',
					],
				],
				'default'   => 'space-between',
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-bottom, {{WRAPPER}} .upk-checkerboard-meta' => 'justify-content: {{VALUE}};',
				],
				'separator' => 'before'
			]
		);

		$this->end_controls_section();

		//New Query Builder Settings
		$this->start_controls_section(
			'section_post_query_builder',
			[
				'label' => __('Query', 'ultimate-post-kit-pro'),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'item_limit',
			[
				'label'     => esc_html__('Item Limit', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 1,
						'max'  => 20,
					],
				],
				'default'   => [
					'size' => 6,
				],
			]
		);

		$this->register_query_builder_controls();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_content_additional',
			[
				'label' => esc_html__('Additional', 'ultimate-post-kit-pro'),
			]
		);

		//Global Title Controls
		$this->register_title_controls();

		$this->add_control(
			'show_category',
			[
				'label'   => esc_html__('Show Category', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
			]
		);

		// $this->add_control(
		// 	'show_like',
		// 	[
		// 		'label'   => esc_html__('Show Like', 'ultimate-post-kit-pro'),
		// 		'type'    => Controls_Manager::SWITCHER,
		// 		'default' => 'yes',
		// 	]
		// );

		// $this->add_control(
		// 	'show_divider',
		// 	[
		// 		'label'   => esc_html__('Show Divider', 'ultimate-post-kit-pro'),
		// 		'type'    => Controls_Manager::SWITCHER,
		// 		'default' => 'yes',
		// 	]
		// );

		$this->add_control(
			'show_excerpt',
			[
				'label'   => esc_html__('Show Text', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'excerpt_length',
			[
				'label'       => esc_html__('Text Limit', 'ultimate-post-kit-pro'),
				'description' => esc_html__('It\'s just work for main content, but not working with excerpt. If you set 0 so you will get full main content.', 'ultimate-post-kit-pro'),
				'type'        => Controls_Manager::NUMBER,
				'default'     => 15,
				'condition'   => [
					'show_excerpt' => 'yes'
				],
			]
		);

		$this->add_control(
			'strip_shortcode',
			[
				'label'     => esc_html__('Strip Shortcode', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'show_meta_author',
			[
				'label'   => esc_html__('Show Author', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
			]
		);


		//Global Date Controls
		$this->register_date_controls();

		//Global Reading Time Controls
		$this->register_reading_time_controls();

		//meta separator control
		$this->add_control(
			'meta_separator',
			[
				'label'     => esc_html__('Meta Separator', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::TEXT,
				'default'   => '|',
				'condition' => [
					'show_date'   => 'yes',
				],
			]
		);
		


		$this->add_control(
			'show_readmore',
			[
				'label' => esc_html__('Read more', 'ultimate-post-kit-pro'),
				'type'  => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'readmore_text',
			[
				'label'       => __('Readmore Text', 'ultimate-post-kit-pro'),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__('Explore', 'ultimate-post-kit-pro'),
				'label_block' => false,
				'condition' => [
					'show_readmore' => 'yes'
				]
			]
		);

		$this->add_control(
			'show_pagination',
			[
				'label'     => esc_html__('Pagination', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SWITCHER,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'global_link',
			[
				'label'        => __('Item Wrapper Link', 'ultimate-post-kit-pro'),
				'type'         => Controls_Manager::SWITCHER,
				'prefix_class' => 'upk-global-link-',
				'description'  => __('Be aware! When Item Wrapper Link activated then title link and read more link will not work', 'ultimate-post-kit-pro'),
			]
		);

		$this->end_controls_section();

		//Style

		$this->start_controls_section(
			'upk_section_style',
			[
				'label' => esc_html__('Items', 'ultimate-post-kit-pro'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->start_controls_tabs(
			'style_items_tabs'
		);

		$this->start_controls_tab(
			'style_item_tab',
			[
				'label' => esc_html__( 'Item', 'ultimate-post-kit-pro' ),
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'item_background',
				'selector'  => '{{WRAPPER}} .upk-checkerboard-item',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'item_border',
				'label'    => esc_html__('Border', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-checkerboard-item',
			]
		);

		$this->add_responsive_control(
			'item_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'item_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'item_margin',
			[
				'label'      => esc_html__('Margin', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);


		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'item_box_shadow',
				'selector' => '{{WRAPPER}} .upk-checkerboard-item',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_item_content_tab',
			[
				'label' => esc_html__( 'Content', 'ultimate-post-kit-pro' ),
			]
		);

		// content padding

		$this->add_responsive_control(
			'content_odd_padding',
			[
				'label'      => esc_html__('Padding (ODD)', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-item:nth-child(odd) .upk-checkerboard-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);

		$this->add_responsive_control(
			'content_even_padding',
			[
				'label'      => esc_html__('Padding (EVEN)', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-item:nth-child(even) .upk-checkerboard-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);



		$this->end_controls_tab();

		$this->end_controls_tabs();

		

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_image',
			[
				'label' => esc_html__('Image', 'ultimate-post-kit-pro'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		// border group control
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'image_border',
				'label'    => esc_html__('Border', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-checkerboard-img',
			]
		);

		$this->add_responsive_control(
			'image_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator' => 'before'
			]
		);

		// box shadow group control
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'image_box_shadow',
				'selector' => '{{WRAPPER}} .upk-checkerboard-img',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_title',
			[
				'label'     => esc_html__('Title', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_title' => 'yes',
				],
			]
		);

		$this->add_control(
			'title_style',
			[
				'label'   => esc_html__('Style', 'ultimate-post-kit-pro'),
				'type'    => Controls_Manager::SELECT,
				'default' => 'underline',
				'options' => [
					'underline'        => esc_html__('Underline', 'ultimate-post-kit-pro'),
					'middle-underline' => esc_html__('Middle Underline', 'ultimate-post-kit-pro'),
					'overline'         => esc_html__('Overline', 'ultimate-post-kit-pro'),
					'middle-overline'  => esc_html__('Middle Overline', 'ultimate-post-kit-pro'),
				],
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard .upk-title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_hover_color',
			[
				'label'     => esc_html__('Hover Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard .upk-title a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name'     => 'title_text_shadow',
				'label'    => __('Text Shadow', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-checkerboard .upk-title',
			]
		);

		$this->add_responsive_control(
			'title_margin',
			[
				'label'      => esc_html__('Margin', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'condition'  => [
					'show_title' => 'yes',
				],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard .upk-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-checkerboard .upk-title',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_text',
			[
				'label'     => esc_html__('Text', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'text_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard .upk-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'text_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-checkerboard .upk-text',
			]
		);

		$this->add_responsive_control(
			'text_margin',
			[
				'label'      => __('Margin', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard .upk-text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_category',
			[
				'label'     => esc_html__('Category', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_category' => 'yes',
				],
			]
		);

		$this->start_controls_tabs('tabs_category_style');

		$this->start_controls_tab(
			'tab_category_normal',
			[
				'label' => esc_html__('Normal', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'category_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-category a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'category_background',
				'selector' => '{{WRAPPER}} .upk-checkerboard-category a',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'category_border',
				'selector' => '{{WRAPPER}} .upk-checkerboard-category a',
			]
		);

		$this->add_responsive_control(
			'category_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-category a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-category a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		// margin
		$this->add_responsive_control(
			'category_margin',
			[
				'label'      => esc_html__('Margin', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-head-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);

		$this->add_responsive_control(
			'category_spacing',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
						'step' => 2,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-category' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'category_shadow',
				'selector' => '{{WRAPPER}} .upk-checkerboard-category a',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'category_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-checkerboard-category a',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_category_hover',
			[
				'label' => esc_html__('Hover', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'category_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-category a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'category_hover_background',
				'selector' => '{{WRAPPER}} .upk-checkerboard-category a:hover',
			]
		);

		$this->add_control(
			'category_hover_border_color',
			array(
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'condition' => array(
					'category_border_border!' => '',
				),
				'selectors' => array(
					'{{WRAPPER}} .upk-checkerboard-category a:hover' => 'border-color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		//meta
		$this->start_controls_section(
			'section_style_meta',
			[
				'label'      => esc_html__('Meta', 'ultimate-post-kit-pro'),
				'tab'        => Controls_Manager::TAB_STYLE,
				'conditions' => [
					'relation' => 'or',
					'terms'    => [
						[
							'name'  => 'show_meta_author',
							'value' => 'yes'
						],
						[
							'name'  => 'show_date',
							'value' => 'yes'
						],
						[
							'name'  => 'show_reading_time',
							'value' => 'yes'
						],
					]
				],
			]
		);
		//author date time tabs
		$this->start_controls_tabs(
			'meta_author_style_tabs'
		);
		//author tab
		$this->start_controls_tab(
			'style_meta_author_text_tab',
			[
				'label' => esc_html__( 'Author', 'ultimate-post-kit-pro' ),
				'condition' => [
					'show_meta_author' => 'yes',
				],
			]
		);
		//author text color
		$this->add_control(
			'meta_author_text_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-author' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_meta_author' => 'yes',
				],
			]
		);
		//author text hover color
		$this->add_control(
			'meta_author_text_hover_color',
			[
				'label'     => esc_html__('Hover Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-author:hover' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_meta_author' => 'yes',
				],
			]
		);
		//author text spacing
		$this->add_responsive_control(
			'meta_author_text_spacing',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-author' => 'gap: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'show_meta_author' => 'yes',
				],
			]
		);
		//author text typography
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'meta_author_text_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-checkerboard-author',
			]
		);

		//heading avatar
		$this->add_control(
			'heading_author_avatar',
			[
				'label'     => esc_html__('Avatar', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::HEADING,
				'condition' => [
					'show_meta_author' => 'yes',
				],
				'separator' => 'before'
			]
		);
		//avatar size
		$this->add_responsive_control(
			'meta_author_img_size',
			[
				'label'      => esc_html__('Avatar Size', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-author img' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'show_meta_author' => 'yes',
				],
			]
		);
		//avatar border
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'meta_author_img_border',
				'label'    => esc_html__('Border', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-checkerboard-author img',
				'condition' => [
					'show_meta_author' => 'yes',
				],
			]
		);
		//avatar border radius
		$this->add_responsive_control(
			'meta_author_img_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-author img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'show_meta_author' => 'yes',
				],
			]
		);
		//author tab end
		$this->end_controls_tab();
		//date tab
		$this->start_controls_tab(
			'style_meta_date_text_tab',
			[
				'label' => esc_html__( 'Date/Time', 'ultimate-post-kit-pro' ),
				'conditions' => [
					'relation' => 'or',
					'terms'    => [
						[
							'name'  => 'show_date',
							'value' => 'yes'
						],
						[
							'name'  => 'show_reading_time',
							'value' => 'yes'
						],
					]
				],
			]
		);
		//date text color
		$this->add_control(
			'meta_date_text_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-date-reading' => 'color: {{VALUE}};',
				],
				'conditions' => [
					'relation' => 'or',
					'terms'    => [
						[
							'name'  => 'show_date',
							'value' => 'yes'
						],
						[
							'name'  => 'show_reading_time',
							'value' => 'yes'
						],
					]
				],
			]
		);
		//typography
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'meta_date_text_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-date-reading',
				'conditions' => [
					'relation' => 'or',
					'terms'    => [
						[
							'name'  => 'show_date',
							'value' => 'yes'
						],
						[
							'name'  => 'show_reading_time',
							'value' => 'yes'
						],
					]
				],
			]
		);
		//date text spacing
		$this->add_responsive_control(
			'meta_date_text_spacing',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-meta' => 'gap: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .upk-checkerboard-meta .upk-reading-time:before' => 'margin: 0 {{SIZE}}{{UNIT}};',
				],
				'conditions' => [
					'relation' => 'or',
					'terms'    => [
						[
							'name'  => 'show_date',
							'value' => 'yes'
						],
						[
							'name'  => 'show_reading_time',
							'value' => 'yes'
						],
					]
				],
			]
		);
		//date tab end
		$this->end_controls_tab();
		//author date time tabs end
		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_link_button',
			[
				'label'     => esc_html__('Read More', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_readmore' => 'yes',
				],
			]
		);

		// icon alignment
		$this->add_responsive_control(
			'link_button_alignment',
			[
				'label'        => esc_html__('Icon Direction', 'ultimate-post-kit-pro'),
				'type'         => Controls_Manager::CHOOSE,
				'options'      => [
					'row'   => [
						'title' => esc_html__('Left', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-h-align-left',
					],
					'row-reverse'  => [
						'title' => esc_html__('Right', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-h-align-right',
					],
				],
				'selectors'    => [
					'{{WRAPPER}} .upk-checkerboard-link a' => 'flex-direction: {{VALUE}};',
				],
				'default'      => 'row-reverse',
				'toggle'       => false,
			]
		);

		$this->start_controls_tabs('tabs_link_button_style');

		$this->start_controls_tab(
			'tab_link_button_normal',
			[
				'label' => esc_html__('Normal', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'link_button_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-link a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'link_button_background',
				'selector' => '{{WRAPPER}} .upk-checkerboard-link a',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'link_button_border',
				'selector' => '{{WRAPPER}} .upk-checkerboard-link a',
			]
		);

		$this->add_responsive_control(
			'link_button_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-link a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'link_button_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-link a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'link_button_spacing',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
						'step' => 2,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-link a' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'link_button_shadow',
				'selector' => '{{WRAPPER}} .upk-checkerboard-link a',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'link_button_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-checkerboard-link a',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_link_button_hover',
			[
				'label' => esc_html__('Hover', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'link_button_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-link a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'link_button_hover_background',
				'selector' => '{{WRAPPER}} .upk-checkerboard-link a:hover',
			]
		);

		$this->add_control(
			'link_button_hover_border_color',
			array(
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'condition' => array(
					'link_button_border_border!' => '',
				),
				'selectors' => array(
					'{{WRAPPER}} .upk-checkerboard-link a:hover' => 'border-color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();
		
		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_like_button',
			[
				'label'     => esc_html__('Like Button', 'ultimate-post-kit-pro'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_like' => 'yes',
				],
			]
		);

		// icon alignment
		$this->add_responsive_control(
			'like_button_alignment',
			[
				'label'        => esc_html__('Direction', 'ultimate-post-kit-pro'),
				'type'         => Controls_Manager::CHOOSE,
				'options'      => [
					'row'   => [
						'title' => esc_html__('Left', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-h-align-left',
					],
					'row-reverse'  => [
						'title' => esc_html__('Right', 'ultimate-post-kit-pro'),
						'icon'  => 'eicon-h-align-right',
					],
				],
				'selectors'    => [
					'{{WRAPPER}} .upk-checkerboard-like-wrap a' => 'flex-direction: {{VALUE}};',
				],
				'default'      => 'left',
			]
		);


		$this->add_responsive_control(
			'like_button_spacing',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
						'step' => 2,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-like-wrap a' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs('tabs_like_button_style');

		$this->start_controls_tab(
			'tab_like_button_normal',
			[
				'label' => esc_html__('Normal', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'like_button_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-like-wrap .upk-like-icon' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'like_button_background',
				'selector' => '{{WRAPPER}} .upk-checkerboard-like-wrap .upk-like-icon',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'like_button_border',
				'selector' => '{{WRAPPER}} .upk-checkerboard-like-wrap .upk-like-icon',
			]
		);

		$this->add_responsive_control(
			'like_button_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-like-wrap .upk-like-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'like_button_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-like-wrap .upk-like-icon' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'like_button_shadow',
				'selector' => '{{WRAPPER}} .upk-checkerboard-like-wrap .upk-like-icon',
			]
		);

		// like button icon size
		$this->add_responsive_control(
			'like_button_icon_size',
			[
				'label'      => esc_html__('Icon Size', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-like-wrap .upk-like-icon' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_like_button_hover',
			[
				'label' => esc_html__('Hover', 'ultimate-post-kit-pro'),
			]
		);

		$this->add_control(
			'like_button_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-like-wrap .upk-like-icon:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'like_button_hover_background',
				'selector' => '{{WRAPPER}} .upk-checkerboard-like-wrap .upk-like-icon:hover',
			]
		);

		$this->add_control(
			'like_button_hover_border_color',
			array(
				'label'     => esc_html__('Border Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'condition' => array(
					'like_button_border_border!' => '',
				),
				'selectors' => array(
					'{{WRAPPER}} .upk-checkerboard-like-wrap .upk-like-icon:hover' => 'border-color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		// heading
		$this->add_control(
			'like_button_count_heading',
			[
				'label'     => esc_html__('Like Count', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		// like button count color
		$this->add_control(
			'like_button_count_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-like-wrap .upk-checkerboard-like' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'like_button_count_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit-pro'),
				'selector' => '{{WRAPPER}} .upk-checkerboard-like-wrap .upk-checkerboard-like',
			]
		);

		// sperator heading
		$this->add_control(
			'separator_heading',
			[
				'label'     => esc_html__('Divider', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'show_divider' => 'yes',
				],
			]
		);

		// separator
		$this->add_control(
			'separator_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-divider' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'show_divider' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'separator_height',
			[
				'label'      => esc_html__('Height', 'ultimate-post-kit-pro'),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'selectors'  => [
					'{{WRAPPER}} .upk-checkerboard-divider' => 'height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'show_divider' => 'yes',
				],
			]
		);

		// spece between
		$this->add_responsive_control(
			'separator_spacing',
			[
				'label'     => esc_html__('Space Between', 'ultimate-post-kit-pro'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
						'step' => 2,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-checkerboard-head-content' => 'gap: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'show_divider' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		//Global Pagination Controls
		$this->register_pagination_controls();
	}

	/**
	 * Main query render for this widget
	 * @param $posts_per_page number item query limit
	 */
	public function query_posts($posts_per_page) {

		$default = $this->getGroupControlQueryArgs();
		if ($posts_per_page) {
			$args['posts_per_page'] = $posts_per_page;
			$args['paged']  = max(1, get_query_var('paged'), get_query_var('page'));
		}
		$args         = array_merge($default, $args);
		$this->_query = new WP_Query($args);
	}

	public function render_category() {

		if (!$this->get_settings('show_category')) {
			return;
		}
		?>
			<div class="upk-checkerboard-category">
				<?php echo upk_get_category($this->get_settings('posts_source')); ?>
			</div>
		<?php
	}


	public function render_author() {

		if (!$this->get_settings('show_meta_author')) {
			return;
		}
		?>
			<a class="upk-checkerboard-author" href="<?php echo get_author_posts_url(get_the_author_meta('ID')) ?>">
				<?php echo get_avatar(get_the_author_meta('ID'), 100); ?>
				<span><?php echo get_the_author() ?></span>
			</a>
		<?php
	}

	public function render_like() {

		if (!$this->get_settings('show_like')) {
			return;
		}
		?>
		<div class="upk-checkerboard-like-wrap">
			<a href="#">
				<i class="upk-icon-heart-full upk-like-icon"></i>
				<span class="upk-checkerboard-like">50</span>
			</a>
		</div>
		<?php
	}

	public function render_divider() {

		if (!$this->get_settings('show_divider')) {
			return;
		}
		?>
      	<span class="upk-checkerboard-divider"></span>
		<?php
	}

	public function render_post_grid_item($post_id, $image_size, $excerpt_length) {
		$settings = $this->get_settings_for_display();

		if ('yes' == $settings['global_link']) {

			$this->add_render_attribute('item', 'onclick', "window.open('" . esc_url(get_permalink()) . "', '_self')", true);
		}

		$this->add_render_attribute('item', 'class', 'upk-checkerboard-item', true);

		?>
		<div <?php $this->print_render_attribute_string('item'); ?>>

			<div class="upk-checkerboard-img">
				<?php $this->render_image(get_post_thumbnail_id($post_id), $image_size); ?>
			</div>

			<div class="upk-checkerboard-content">
				<div class="upk-checkerboard-head-content">
			    	<?php $this->render_category(); ?>
					<!-- TODO -->
					<?php // $this->render_divider(); ?>
					<?php // $this->render_like(); ?>
				</div>

				<?php $this->render_title(substr($this->get_name(), 4)); ?>
				<?php $this->render_excerpt($excerpt_length); ?>

				<div class="upk-checkerboard-bottom">
					<div class="upk-checkerboard-meta">
						<?php $this->render_author(); ?>
			
						<?php if ($settings['show_date'] or $settings['show_reading_time']) : ?>
						<div class="upk-date-reading upk-flex upk-flex-middle">
							<?php $this->render_date(); ?>
							<?php if ('yes' === $settings['show_reading_time']) : ?>
								<div class="upk-reading-time" data-separator="<?php echo esc_html($settings['meta_separator']); ?>">
									<?php echo ultimate_post_kit_reading_time(get_the_content(), $settings['avg_reading_speed']); ?>
								</div>
							<?php endif; ?>
						</div>
					<?php endif; ?>

					</div>

					<?php if ($settings['show_readmore'] === 'yes' and !empty($settings['readmore_text'])) : ?>
						<div class="upk-checkerboard-link">
							<a href="<?php echo esc_url(get_permalink()); ?>">
								<i class="upk-icon-arrow-right-8"></i>
								<span><?php echo esc_html($settings['readmore_text']); ?></span>
							</a>
						</div>
			    	<?php endif; ?>
				</div>
			</div>
		</div>

	
	<?php
	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->query_posts($settings['item_limit']['size']);

		$wp_query = $this->get_query();

		if (!$wp_query->found_posts) {
			return;
		}

	?>
		<div class="upk-checkerboard upk-checkerboard-<?php echo esc_attr($settings['preset_style']) ?>">
			<?php while ($wp_query->have_posts()) :
				$wp_query->the_post();

				$thumbnail_size = $settings['primary_thumbnail_size'];

			?>

				<?php $this->render_post_grid_item(get_the_ID(), $thumbnail_size, $settings['excerpt_length']); ?>

			<?php endwhile; ?>
		</div>

		<?php

		if ($settings['show_pagination']) { ?>
			<div class="ep-pagination">
				<?php ultimate_post_kit_post_pagination($wp_query, $this->get_id()); ?>
			</div>
		<?php
		}
		wp_reset_postdata();
	}
}
