<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://https://theeventprime.com
 * @since      1.0.0
 *
 * @package    Eventprime_Zapier_Integration
 * @subpackage Eventprime_Zapier_Integration/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Eventprime_Zapier_Integration
 * @subpackage Eventprime_Zapier_Integration/public
 * @author     EventPrime <support@metagauss.com>
 */
class Eventprime_Zapier_Integration_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */

	private $url;
    private $api_key;
    private $dao;

	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Eventprime_Zapier_Integration_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Eventprime_Zapier_Integration_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/eventprime-zapier-integration-public.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Eventprime_Zapier_Integration_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Eventprime_Zapier_Integration_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/eventprime-zapier-integration-public.js', array( 'jquery' ), $this->version, false );

	}

	public function ep_zapier_api_endpoint(){

        if ( ! isset( $_GET['eventprime-integration'] ) || $_GET['eventprime-integration'] !== 'zapier' ) 
        {
            return;
        }
        

        // Parse data and get site API key
        $request = array_map( 'sanitize_text_field', $_GET );
        $request = array_map( 'trim', $request );
        $payload = file_get_contents( 'php://input' );
        $payload = ! empty( $payload ) ? json_decode( $payload, true ) : array();
        array_walk_recursive( $payload, function( &$value ) {
            $value = sanitize_text_field( $value );
        });
        
        // Validate Site URL format
        if ( ! empty( $request['site_url'] ) ) {
            if ( ! preg_match( '/\/$/', $request['site_url'] ) ) {
                $this->bail( __( 'Bad request: site URL needs to have trailing slash at the end of the URL.', 'eventprime-zapier-integration' ), 400 );
            }
        }


        // Authenticate
        if ( ! $this->is_authenticated( $request ) ) {
            $this->bail( __( 'Invalid API key', 'eventprime-zapier-integration' ), 401 );
        }

        if ( isset( $request['action'] ) ) {
            switch ( $request['action'] ) {
                case 'subscribe':
                    $response = $this->init_action_subscribe($request);
                    $this->bail($response);
                break;

                case 'perform':
                    $response = $this->init_action_perform($request);
                    $this->bail($response);
                break;

                case 'unsubscribe':
                    $response = $this->init_action_unsubscribe($request);
                break;

                //Actions
                case 'init_action':
                    $response = $this->init_action( $request['action_key']);
                    $this->bail( $response );
                break;

                case 'create_event':
                    $response = $this->init_action_create_event($request);
                    $this->bail( $response );
                break;

                case 'create_booking':
                    $response = $this->init_action_create_booking($request);
                    $this->bail( $response );
                break;

                default:
                    $this->bail( __( 'Unknown request parameters', 'eventprime-zapier-integration' ), 404 );
                break;
            }
        }
        $site_title = get_bloginfo( 'name' );
        $msg = array('msg'=>'Success','site_title'=>$site_title);
        $this->bail( $msg, 200 );


    }

    public function init_action_perform($request)
    {
        $data = $this->generate_sample_data($request);
        $data = $this->normalize_response($data);

        return $data;

    }

    public function init_action( $action_key) {
        $response = false;
        return apply_filters( 'eventprime_zapier_api_action_payload', $response, $action_key );
    }

    public function generate_sample_data($request){
        $triggers = $request['trigger'];

        $Zapier_request = new Eventprime_Zapier_Requests;
        switch ( $triggers ) 
        {
            case 'create_event':
            case 'update_event':
            case 'delete_event':
                $data = $Zapier_request->all_events_data($triggers);
            break;

            case 'all_events':
               
                $args = array(
                    'numberposts'      => -1,
                    'post_type'        => 'em_event',
                    'post_status'    => 'publish'
                );
                $events = get_posts($args);
                $data = array();
                if(!empty($events)){
                    foreach($events as $event){
                        $data[] = array(
                            'event_id' => $event->ID,         
                            'event_name' => $event->post_title 
                        );
                    }
                }
            break;

            case 'get_tickets_by_event':
                $data = $Zapier_request->ep_get_tickets_by_event($request);
            break;

            case 'create_venue':
            case 'update_venue':
            case 'delete_venue':
                $data = $Zapier_request->all_venues_data();
            break;

            case 'create_organizer':
            case 'update_organizer':
            case 'delete_organizer':
                $data = $Zapier_request->all_organizers_data();
            break;

            case 'create_performer':
            case 'update_performer':
            case 'delete_performer':
                $data = $Zapier_request->all_performers_data($triggers);
            break;

            case 'confirm_booking':
                $data = $Zapier_request->all_bookings_data('completed');
            break;

            case 'pending_booking':
                $data = $Zapier_request->all_bookings_data('pending');
            break;

            case 'cancel_booking':
                $data = $Zapier_request->all_bookings_data('cancelled');
            break;

            case 'refund_booking':
                $data = $Zapier_request->all_bookings_data('refunded');
            break;

            case 'failed_booking':
                $data = $Zapier_request->all_bookings_data('failed');
            break;

            default:
                $data = array(
                    'msg' => esc_html__( 'No such triggers available. Please select a valid trigger to proceed.', 'eventprime-zapier-integration' )
                );
            break;
        }

        return $data;
    }

    public function init_action_unsubscribe($request)
    {
        $data = array(
            'name'     => $request['zapId'],
            'type'   => $request['trigger'],
            'webhook_url' => $request['hookUrl']
        );
        $this->delete_zapier_subscription($request['zapId']);
        return $data;
    }

    public function delete_zapier_subscription($zapId){
        $ep_dbhandler = new EP_DBhandler(); 
        $ep_dbhandler->remove_row( 'Zapier_Integration', 'name', $zapId ); 
    }
     
     public function init_action_subscribe($request)
     {
          
        $data = array(
            'name'     => $request['zapId'],
            'type'   => $request['trigger'],
            'webhook_url' => $request['hookUrl']
        );

        $dbhandler = new EP_DBhandler;
        $identifier = 'Zapier_Integration';    
        $dbhandler->insert_row($identifier, $data); 

        $response = $this->generate_sample_data($request);

        $this->init_trigger($request['trigger'], $response);
        return $data;
    }
    
    public function init_trigger($type,$response)
    {
        $dbhandler = new EP_DBhandler;
        $where = array('type'=>$type);
        $triggers = $dbhandler->get_all_result('Zapier_Integration' ,'*', $where);
        if(!empty($triggers)):
            foreach($triggers as $trigger)
            {
                $this->send_trigger($trigger,$response,$type);
            }
        endif;
    }
    
    public function send_trigger( $trigger,$response,$type ) {
       
        $response = apply_filters( 'eventprime_zapier_api_response', $response, $trigger,$type );
        $response = $this->normalize_response($response);

        $response = wp_remote_post( $trigger->webhook_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'charset' => 'utf-8'
            ),
            'body' => wp_json_encode( $response ),
        ));

    }   
        
    private function normalize_response($data) {
        // Objects → arrays
        if (is_object($data)) {
            $data = (array) $data;
        }

        // Arrays: recurse and preserve keys
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = $this->normalize_response($value);
            }
            return $data;
        }

        // Non-string scalars: passthrough (int/float/bool/null)
        if (!is_string($data)) {
            return $data;
        }

        $original = $data;

        // 1) Try WordPress maybe_unserialize (guard if not available)
        if (function_exists('maybe_unserialize')) {
            $maybe = maybe_unserialize($data);
            if ($maybe !== $data) {
                return $this->normalize_response($maybe);
            }
        }

        // Helper to try JSON decode (arrays/objects only), tolerant of invalid UTF-8
        $try_json = static function (string $s) {
            $v = json_decode($s, true, 512, JSON_INVALID_UTF8_SUBSTITUTE);
            return (json_last_error() === JSON_ERROR_NONE && (is_array($v) || is_object($v))) ? $v : null;
        };

        // 2) Direct JSON decode (covers both {...} and [...])
        if (($decoded = $try_json($data)) !== null) {
            return $this->normalize_response($decoded);
        }

        // 3) Unslash then JSON (handles \" and \\ from WP/Zapier)
        $unslashed = function_exists('wp_unslash') ? wp_unslash($data) : stripcslashes($data);
        if ($unslashed !== $data) {
            if (($decoded = $try_json($unslashed)) !== null) {
                return $this->normalize_response($decoded);
            }
        }

        // 4) Quoted JSON: '"{...}"' or '"[...]"'
        $trim = trim($data);
        if ($trim !== '' && $trim[0] === '"' && substr($trim, -1) === '"') {
            $inner = substr($trim, 1, -1);
            if (($decoded = $try_json($inner)) !== null) {
                return $this->normalize_response($decoded);
            }
            $inner2 = function_exists('wp_unslash') ? wp_unslash($inner) : stripcslashes($inner);
            if ($inner2 !== $inner && ($decoded = $try_json($inner2)) !== null) {
                return $this->normalize_response($decoded);
            }
        }

        // Nothing decoded → return original string untouched
        return $original;
    }

    public function is_authenticated( $request ) {
		$ep_functions = new Eventprime_Basic_Functions(); 
         $this->api_key = $ep_functions->ep_get_global_settings('zapier_api_key');
               
        if ( isset( $request['api_key'] ) && strtoupper( $request['api_key'] ) === strtoupper( $this->api_key) ) {

            return true;
        } else {
            return false;
        }
    }

    public function bail( $response = '', $status_code = 200 ) {
        if ( ! is_array( $response ) ) {
            $response = array(
                'status'  => $status_code,
                'message' => $response,
            );
        }
        header( 'Content-Type: application/json;charset=utf-8;' );
        http_response_code( $status_code );
        echo wp_json_encode( $response );
        exit();
    }

    public function init_action_create_booking($request) {
        $event_id       = isset($request['event_id']) ? absint($request['event_id']) : 0;
        $ticket_id      = isset($request['ticket_id']) ? absint($request['ticket_id']) : 0;
        $user_email     = isset($request['email']) ? sanitize_email($request['email']) : '';
        $booking_status = isset($request['booking_status']) ? sanitize_text_field($request['booking_status']) : 'completed';

        $ep_functions = new Eventprime_Basic_Functions;
        $sanitizer    = new EventPrime_sanitizer;

        // 1. Validate required data
        $event_id  = absint($event_id);
        $ticket_id = absint($ticket_id);
        if (empty($event_id) || empty($ticket_id) || empty($user_email)) {
            return array(
                'msg' => esc_html__( 'Missing required booking data.', 'eventprime-zapier-integration' )
            );
        }

        // 2. Check event and ticket
        $ticket_data_object = $ep_functions->ep_get_ticket_data($ticket_id);

        if (empty($ticket_data_object)) {
           return array(
                'msg' => esc_html__( 'Invalid ticket.', 'eventprime-zapier-integration' )
            );
        }

        // 3. Find or create user
        $user = get_user_by('email', $user_email);
        if (empty($user)) {
            $username = sanitize_user(current(explode('@', $user_email)));
            $random_password = wp_generate_password(12, true);
            $user_id = wp_create_user($username, $random_password, $user_email);
        } else {
            $user_id = $user->ID;
        }

		$ticket_price = 0;
		$ticket_name  = '';
		if (!empty($ticket_data_object)) {
			if (is_object($ticket_data_object)) {
				$ticket_price = isset($ticket_data_object->price) ? (float) $ticket_data_object->price : 0;
				$ticket_name  = isset($ticket_data_object->name) ? $ticket_data_object->name : 'Ticket';
			} elseif (is_array($ticket_data_object)) {
				$ticket_price = isset($ticket_data_object['price']) ? (float) $ticket_data_object['price'] : 0;
				$ticket_name  = isset($ticket_data_object['name']) ? $ticket_data_object['name'] : 'Ticket';
			}
		}

		// 2. Build ticket JSON (must match frontend structure)
		$ticket_json = json_encode(array(
			array(
				'id'            => $ticket_id,
				'category_id'   => "0",
				'name'          => $ticket_name,
				'price'         => $ticket_price,
				'qty'           => 1,
				'offer'         => 0,
				'additional_fee'=> array(),
				'subtotal'      => $ticket_price,
			)
		));

		// 3. Build $data array same as your log
		$data = array(
			'ep_event_booking_ticket_data'    => $ticket_json,
			'ep_event_booking_event_id'       => $event_id,
			'ep_event_booking_user_id'        => $user_id,
			'ep_event_booking_sub_total_price'=> $ticket_price,
			'ep_event_booking_total_price'    => $ticket_price,
			'ep_event_booking_total_tickets'  => 1,
			'ep_save_event_booking_nonce'     => wp_create_nonce('ep_save_event_booking'),
			'_wp_http_referer'                => '/booking/', // fake referer, can be blank
			'ep_booking_attendee_fields'      => array(
				1 => array(
					1 => array(
						'name' => array(
							'first_name' => '',
							'last_name'  => '',
						),
						1 => array(
							'label'     => 'User Mail',
							'user_mail' => $user_email,
						),
					),
				),
			),
			'ep_event_booking_event_fixed_price' => 0,
            'ep_booking_guest_booking_field' => array(
                'ep_gb_email' => $user_email
            ),
		);

        // 5. Recalculate and verify
        /*$data = $ep_functions->ep_recalculate_and_verify_the_cart_data($data, array());
		error_log( print_r( $data, true ) );
		//return $data;
        if ($data == 'ticket_sold' || $data === false) {
            return array(
                'msg' => esc_html__( 'Tickets are sold out.', 'eventprime-zapier-integration' )
            );
        }*/

        // 6. Actually insert booking (shortcut from save_event_booking)
        $event_name = get_the_title($event_id);
        $new_post   = array(
            'post_title'  => $event_name,
            'post_status' => $booking_status, // or 'failed' if payment pending
            'post_type'   => 'em_booking',
            'post_author' => $user_id,
        );
        $new_post_id = wp_insert_post($new_post);

        update_post_meta($new_post_id, 'em_id', $new_post_id);
        update_post_meta($new_post_id, 'em_event', $event_id);
        update_post_meta($new_post_id, 'em_date', current_time('timestamp', true));
        update_post_meta($new_post_id, 'em_user', $user_id);
        update_post_meta($new_post_id, 'em_name', $event_name);
        update_post_meta($new_post_id, 'em_status', $booking_status);
        update_post_meta($new_post_id, 'em_payment_method', 'none');

        $order_info = array(
            'tickets'           => json_decode($data['ep_event_booking_ticket_data']),
            'event_fixed_price' => 0,
            'booking_total'     => (float)$data['ep_event_booking_total_price'],
        );
        update_post_meta($new_post_id, 'em_order_info', $order_info);

		$booking_controller      = new EventPrime_Bookings;
		$booking_controller->update_status($new_post_id, $booking_status);

        return array(
            'booking_id' => $new_post_id,
            'event_id'   => $event_id,
            'user_id'    => $user_id,
            'msg' => esc_html__( 'Booking created successfully.', 'eventprime-zapier-integration' ),
        );
    }

    public function init_action_create_event( $request ){
            
        $ep_functions = new Eventprime_Basic_Functions(); 
        
        $event_data = new stdClass();
        $title                 = isset($request['em_name']) ? sanitize_text_field($request['em_name']) : '';
        $description           = isset($request['description']) ? sanitize_textarea_field($request['description']) : '';
        $start_date = isset($request['em_start_date']) ? (new DateTime($request['em_start_date']))->format('Y-m-d') : '';
        $start_time = isset($request['em_start_date']) ? (new DateTime($request['em_start_date']))->format('h:i A') : '';
        $end_date   = isset($request['em_end_date'])   ? (new DateTime($request['em_end_date']))->format('Y-m-d')   : '';
        $end_time   = isset($request['em_end_date'])   ? (new DateTime($request['em_end_date']))->format('h:i A')   : '';
        $all_day               = isset($request['all_day']) ? absint($request['all_day']) : 0;
        $event_type            = isset($request['event_type']) ? sanitize_text_field($request['event_type']) : '';
        $ticket_price          = isset($request['ticket_price']) ? floatval($request['ticket_price']) : 0;
        $location_name         = isset($request['location_name']) ? sanitize_text_field($request['location_name']) : '';
        $longitude             = isset($request['longitude']) ? sanitize_text_field($request['longitude']) : '';
        $latitude              = isset($request['latitude']) ? sanitize_text_field($request['latitude']) : '';
        $address               = isset($request['address']) ? sanitize_textarea_field($request['address']) : '';
        $seating_capacity      = isset($request['seating_capacity']) ? absint($request['seating_capacity']) : 0;
        $operator              = isset($request['operator']) ? sanitize_text_field($request['operator']) : '';
        $performers            = isset($request['performers']) ? sanitize_text_field($request['performers']) : '';
        $organizer             = isset($request['organizer']) ? sanitize_text_field($request['organizer']) : '';
        $imp_organizer_phones = isset($request['organizer_phones']) ? sanitize_text_field($request['organizer_phones']) : '';
        $imp_organizer_emails = isset($request['organizer_emails']) ? sanitize_text_field($request['organizer_emails']) : '';
        $imp_organizer_websites = isset($request['organizer_websites']) ? sanitize_text_field($request['organizer_websites']) : '';

        if( empty( $title ) ){
            $this->bail(
                esc_html__( "Event name can't be empty.", 'eventprime-zapier-integration' ),
                200
            );
        }
        if( empty( $start_date ) ){
            $this->bail(
                esc_html__( "Event start date can't be empty.", 'eventprime-zapier-integration' ),
                400
            );
        }
        if( empty( $end_date ) ){
            $this->bail(
                esc_html__( "Event end date can't be empty.", 'eventprime-zapier-integration' ),
                200
            );
        }
            if( empty( $start_time ) ){
            $this->bail(
                esc_html__( "Event start time can't be empty.", 'eventprime-zapier-integration' ),
                400
            );
        }
        if( empty( $end_time ) ){
            $this->bail(
                esc_html__( "Event end time can't be empty.", 'eventprime-zapier-integration' ),
                200
            );
        }
        $event_data->name = htmlspecialchars_decode(sanitize_text_field(strip_tags($title)));
        $event_data->description = sanitize_text_field($description);
        $allday = (int) $all_day;

        if(!empty($allday) || !trim($start_date) && !trim($end_date)){
            $allDayDate = $start_date;
            $allday = 1;
            $start_time = "00:00 AM";
            $end_time = "00:00 AM";
            $start_date = $allDayDate;
            $allEndDate = date('m/d/Y', strtotime($allDayDate. ' + 1 days'));
            $end_date = $allEndDate;
        }
        $event_data->em_start_date = !empty($start_date) ? $ep_functions->ep_date_to_timestamp($start_date) : '';
        $event_data->em_end_date = !empty($end_date) ? $ep_functions->ep_date_to_timestamp($end_date): '';
        $event_data->em_start_time = !empty($start_date) && empty($start_time) ? '12:00 AM' : (!empty($start_time) ? $start_time : '');
        $event_data->em_end_time   = !empty($end_date) && empty($end_time) ? '12:00 AM' : (!empty($end_time) ? $end_time : '');

        $ep_date_time_format = 'Y-m-d';
        $event_data->em_start_date_time = $ep_functions->ep_datetime_to_timestamp( $ep_functions->ep_timestamp_to_date( $event_data->em_start_date, 'Y-m-d', 1 ) . ' ' . $start_time, $ep_date_time_format, '', 0, 1 ); 
        $event_data->em_end_date_time = $ep_functions->ep_datetime_to_timestamp( $ep_functions->ep_timestamp_to_date( $event_data->em_end_date, 'Y-m-d', 1 ) . ' ' . $end_time, $ep_date_time_format, '', 0, 1 );

       $event_data->em_event_text_color = '#000000';

        $event_data->em_ticket_price = $ticket_price;
        // event venue
        $location_term = new stdClass();
        if( ! empty( $ticket_price ) ){
            $event_data->em_enable_booking = 'bookings_on';
        }
        if( ! empty( trim( $location_name ) ) ){
            $location_term = get_term_by( 'name', $location_name, 'em_venue' );
        }
        else{
            if( ! empty( trim( $address ) )){
                $location_exp = explode(",", $address);
                $location_name = $location_exp[0];
                $location_term = get_term_by('name', $location_name, 'em_venue');
            }
        }
        if( empty( $location_term ) && ! empty( $location_name ) ){
            $venue_data = new stdClass();
            $venue_data->name = sanitize_text_field($location_name);
            $venue_data->em_type = 'standings';
            $venue_data->em_address = sanitize_text_field($address);
            if(empty($address)){
                $venue_data->em_address = sanitize_text_field($location_name);
            }
            $venue_data->em_seating_capacity = $seating_capacity;
            $venue_data->em_lng = $longitude;
            $venue_data->em_lat = $latitude;
            $venue_data->em_operator = $operator;
            $event_venue_id = $ep_functions->create_venue((array)$venue_data);
            if(!empty($event_venue_id)){
                $event_data->em_venue = $event_venue_id;
            }
        }
        else{
            $event_venue_id = (isset($location_term->term_id) ? absint($location_term->term_id) : 0);
            $event_data->em_venue = $event_venue_id;
        }
        // event performers
        if( ! empty( $performers ) ){
            $performers = explode(' | ', $performers);
            $all_performers = $ep_functions->get_performer_all_data();
            $performer_ids = array();
            foreach ($performers as $imp_per) {
                    $posts = get_posts([
                        'post_type'  => 'em_performer',
                        'title' => $imp_per,
                    ]);
                    
                    if(!empty($posts) && count($posts) > 0){
                        $performer_ids[] = $posts[0]->ID;
                    }else{
                        $performer_data = new stdClass();
                        $performer_data->name = $imp_per;
                        $performer_data->em_type = 'person';
                        $performer = $ep_functions->insert_performer_post_data((array)$performer_data);
                        $performer_ids[] = $performer;
                        break;
                    }
                
            }
            $event_data->em_performer = $performer_ids;
            if( ! empty( $event_data->em_performer ) ){
                $event_data->em_enable_performer = 1;
            }
        }
        // event organizer
        if(!empty($organizer)){
            $event_organizers = array_filter(explode(" | ", $organizer));
            if(!empty($imp_organizer_phones)){ 
                $organizer_phones = explode(" | ", $imp_organizer_phones); 
                
            }
            if(!empty($imp_organizer_emails)){ 
                $organizer_emails = explode(" | ", $imp_organizer_emails);
            }
            if(!empty($imp_organizer_websites)){ 
                $organizer_websites = explode(" | ", $imp_organizer_websites);
            }
            $organizer_ids = array();
            foreach($event_organizers as $key => $organizer_name){
                $organizer = get_term_by('name',$organizer_name,'em_event_organizer');
                if(!empty($organizer)){
                    $organizer_ids[] = $organizer->term_id;
                    
                }else{
                    $org_data = new stdClass();
                    $org_data->name = $organizer_name;
                    if(!empty($organizer_phones) && isset($organizer_phones[$key]) && !empty($organizer_phones[$key])){
                        $org_data->em_organizer_phones = explode(', ', $organizer_phones[$key]);
                    }
                    if(!empty($organizer_emails) && isset($organizer_emails[$key]) && !empty($organizer_emails[$key])){
                        $org_data->em_organizer_emails = explode(', ', $organizer_emails[$key]);
                    }
                    if(!empty($organizer_websites) && isset($organizer_websites[$key]) && !empty($organizer_websites[$key])){
                        $org_data->em_organizer_websites = explode(', ', $organizer_websites[$key]);
                    }
                    $org_data->em_social_links = array(
                        'facebook'=>'',
                        'linkedin'=>'',
                        'instagram'=>'',
                        'twitter'=>''
                    );
                    $org_id = $ep_functions->create_organizer((array)$org_data);
                
                    if($org_id){
                        $organizer_ids[] = $org_id;
                    }
                }
            }
            $event_data->em_organizer = $organizer_ids;
        }
        // event type
        if( isset( $event_type ) && ! empty( $event_type ) ){
            $eventType = trim($event_type);
            $type_term = get_term_by('name', $eventType, 'em_event_type');

            if(empty($type_term)){
                $type_data = new stdClass();
                $type_data->name = sanitize_text_field($event_type);
                $event_type_id = $ep_functions->create_event_types((array)$type_data);
                if(!empty($event_type_id)){
                    $event_data->em_event_type = $event_type_id;
                }
            }
            else{
                $event_type_id = $type_term->term_id;
                $event_data->em_event_type = $event_type_id;
            }
        }
        
        // insert event data
        $event_id = $ep_functions->insert_event_post_data( (array)$event_data );
        if( ! empty( $event_id ) ){
            update_post_meta($event_id, 'em_start_date_time', $event_data->em_start_date_time);
            update_post_meta($event_id, 'em_end_date_time', $event_data->em_end_date_time);
            update_post_meta( $event_id, 'em_event_text_color', $event_data->em_event_text_color);
            $response = array(
                'msg'  => esc_html__( 'Event successfully created.', 'eventprime-zapier-integration' ),
                'data' => $event_data,
                'ID'   => $event_id
            );
        }
        else{
            $response = array(
                'msg' => esc_html__( 'Something went wrong.', 'eventprime-zapier-integration' ),
                'ID'  => 0
            );
        }
        $this->bail( $response, 200 );
    } 
}

