<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://https://theeventprime.com
 * @since      1.0.0
 *
 * @package    Eventprime_Zapier_Integration
 * @subpackage Eventprime_Zapier_Integration/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Eventprime_Zapier_Integration
 * @subpackage Eventprime_Zapier_Integration/includes
 * @author     EventPrime <support@metagauss.com>
 */
class Eventprime_Zapier_Integration {
	public $current_page;
	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Eventprime_Zapier_Integration_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'EVENTPRIME_ZAPIER_INTEGRATION_VERSION' ) ) {
			$this->version = EVENTPRIME_ZAPIER_INTEGRATION_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'eventprime-zapier-integration';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Eventprime_Zapier_Integration_Loader. Orchestrates the hooks of the plugin.
	 * - Eventprime_Zapier_Integration_i18n. Defines internationalization functionality.
	 * - Eventprime_Zapier_Integration_Admin. Defines all hooks for the admin area.
	 * - Eventprime_Zapier_Integration_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-eventprime-zapier-integration-deactivator.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-eventprime-zapier-integration-activator.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-eventprime-zapier-integration-services.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-eventprime-zapier-integration-helper.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-eventprime-zapier-requests.php';
		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-eventprime-zapier-integration-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-eventprime-zapier-integration-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-eventprime-zapier-integration-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-eventprime-zapier-integration-public.php';

		$this->loader = new Eventprime_Zapier_Integration_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Eventprime_Zapier_Integration_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Eventprime_Zapier_Integration_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Eventprime_Zapier_Integration_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		$this->loader->add_action('admin_notices', $plugin_admin, 'ep_plugin_activation_notice_fun');
		$this->loader->add_action('network_admin_notices', $plugin_admin, 'ep_plugin_activation_notice_fun');

		$this->loader->add_filter( 'ep_add_pages_options', $plugin_admin,'add_pages_setting_meta' , 10, 1 );
        // view settings
        $this->loader->add_filter( 'ep_admin_settings_tabs', $plugin_admin,'add_zapier_settings_tab', 10, 2 );
        $this->loader->add_filter( 'ep_extensions_settings', $plugin_admin,'ep_extensions_settings', 10, 1 ); 
        $this->loader->add_action( 'ep_get_extended_settings_tabs_content', $plugin_admin, 'add_zapier_settings_tab_content' );
        // save settings
        $this->loader->add_action( 'ep_submit_global_setting', $plugin_admin, 'save_zapier_settings' );
        // add global settings options
		$this->loader->add_filter( 'ep_add_global_setting_options', $plugin_admin, 'ep_add_zapier_global_setting_options', 10, 2 );
		// ep license module addon
		$this->loader->add_action( 'ep_add_license_setting_blocks', $plugin_admin, 'ep_add_zapier_license_setting_block' );
		$this->loader->add_filter( 'ep_pupulate_license_item_id', $plugin_admin, 'ep_pupulate_zapier_license_item_id' , 10, 2 ); // populate license item id.
		$this->loader->add_filter( 'ep_pupulate_license_item_name', $plugin_admin, 'ep_pupulate_zapier_license_item_name' , 10, 2 ); // populate license item name.
		$this->loader->add_action( 'ep_save_license_settings', $plugin_admin, 'ep_save_zapier_license_setting', 10, 2 );

		// Trigger Event create update hook
		$this->loader->add_action( 'ep_after_save_event_data', $plugin_admin, 'trigger_zapier_event_create_update' , 10, 2 );
		$this->loader->add_action( 'trashed_post', $plugin_admin, 'trigger_zapier_delete' , 10,1 );
		//$this->loader->add_action( 'eventprime_process_zapier_delete', $plugin_admin, 'trigger_zapier_delete' , 10,1 );

		//trigger performer create update hook
		$this->loader->add_action( 'ep_after_save_performer_data', $plugin_admin, 'trigger_zapier_performer_create_update' , 10, 2 );

        $this->loader->add_action( 'ep_after_booking_complete', $plugin_admin, 'trigger_zapier_event_booking' , 10, 2 );
		$this->loader->add_action( 'ep_booking_after_status_updated', $plugin_admin, 'trigger_zapier_event_booking' , 10, 2 );
		$this->loader->add_action( 'ep_after_booking_cancelled', $plugin_admin, 'trigger_zapier_cancel_event_booking' , 10,1 );
		
		// Trigger Organizer create hook
		$this->loader->add_action( 'em_after_organizer_created', $plugin_admin, 'trigger_zapier_create_new_organizer' , 10,2 );
        // Trigger Organizer update hook
		$this->loader->add_action( 'ep_after_edit_organizer_data', $plugin_admin, 'trigger_zapier_update_organizer', 10, 2 );

		// Trigger venue organizer delete hook
		$this->loader->add_action( 'pre_delete_term', $plugin_admin, 'trigger_zapier_delete_taxonomy', 10, 2 );

		$this->loader->add_action( 'ep_after_save_venue_data', $plugin_admin, 'trigger_zapier_venue_create_update', 10,2 );

        $this->current_page = isset($_GET['page']) ? $_GET['page'] : '';
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Eventprime_Zapier_Integration_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

		$this->loader->add_action( 'init', $plugin_public, 'ep_zapier_api_endpoint' );

	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Eventprime_Zapier_Integration_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
